# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2021 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

# Functions common to all cmdlets
Function ValidateGetAdapterNameParams($AdapterName, $Adapters, [ref]$ErrorMessages)
{
    $AdapterNames = @()

    if ((-Not $Adapters) -and $AdapterName)
    {
        foreach ($n in $AdapterName)
        {
            $TmpAdapterNames = @()
            try
            {
                $TmpAdapters = ($script:PnpDevice.Where({ $_.Name -Like $n }))
                if ($TmpAdapters)
                {
                    $TmpAdapterNames = $TmpAdapters.Name
                }
            }
            catch
            {
                # Failed due to Adapters passed to Name parameter
                $AdapterNames = @()
                $ErrorMessages.Value += $Messages.InvalidParams
                break
            }

            if (-Not $TmpAdapterNames)
            {
                $ErrorMessages.Value += $Messages.AdapterNotFound -f $n
                continue
            }
            $AdapterNames += $TmpAdapterNames
        }
    }
    elseif ((-Not $AdapterName) -and $Adapters)
    {
        foreach ($a in $Adapters)
        {
            if (CheckPropertyExists $a "CreationClassName")
            {
                if ($a.CreationClassName -eq "MSFT_NetAdapter")
                {
                    $AdapterNames += $a.ifDesc
                }
                elseif ($a.CreationClassName -eq "Win32_NetworkAdapter" -or
                    $a.CreationClassName -eq "IANet_PhysicalEthernetAdapter" -or
                    $a.CreationClassName -eq "Win32_PnpEntity")
                {
                    $AdapterNames += $a.Name
                }
                else
                {
                    $ErrorMessages.Value += $Messages.InvalidObject -f $a.GetType().Name
                }
            }
            elseif ($null -ne $a.PSTypeNames -and $a.PSTypeNames[0] -eq "IntelEthernetAdapter")
            {
                $AdapterNames += $a.Name
            }
            else
            {
                $ErrorMessages.Value += $Messages.InvalidObject -f $a.GetType().Name
            }
        }
    }
    elseif (-Not ($AdapterName -and $Adapters))
    {
        $AdapterNames = ($script:PnpDevice.Where({ $_.Manufacturer -eq "Intel" })).Name
    }
    elseif ($AdapterName -and $Adapters)
    {
        $ErrorMessages.Value += $Messages.InvalidParamsAdapterAndName
    }

    return $AdapterNames
}

Function ValidateSetAdapterNameParams($AdapterName, $Adapters, [ref]$ErrorMessages)
{
    $AdapterNames = @()

    do
    {
        if ($AdapterName -and $Adapters)
        {
            $ErrorMessages.Value += $Messages.InvalidParamsAdapterAndName
            break
        }
        elseif ($AdapterName)
        {
            foreach ($n in $AdapterName)
            {
                $TmpAdapterNames = @()
                try
                {
                    $TmpAdapters = ($script:PnpDevice.Where({ $_.Name -Like $n }))
                    if ($TmpAdapters)
                    {
                        $TmpAdapterNames = $TmpAdapters.Name
                    }
                }
                catch
                {
                    # Failed due to Adapters passed to Name parameter
                    $AdapterNames = @()
                    $ErrorMessages.Value += $Messages.InvalidParams
                    break
                }

                if (-Not $TmpAdapterNames)
                {
                    $ErrorMessages.Value += $Messages.AdapterNotFound -f $n
                    continue
                }
                $AdapterNames += $TmpAdapterNames
            }
        }
        elseif ($Adapters)
        {
            foreach ($a in $Adapters)
            {
                if (CheckPropertyExists $a "CreationClassName")
                {
                    if ($a.CreationClassName -eq "MSFT_NetAdapter")
                    {
                        $AdapterNames += $a.ifDesc
                    }
                    elseif ($a.CreationClassName -eq "Win32_NetworkAdapter" -or
                        $a.CreationClassName -eq "IANet_PhysicalEthernetAdapter" -or
                        $a.CreationClassName -eq "Win32_PnpEntity")
                    {
                        $AdapterNames += $a.Name
                    }
                    else
                    {
                        $ErrorMessages.Value += $Messages.InvalidObject -f $a.GetType().Name
                    }
                }
                elseif ($null -ne $a.PSTypeNames -and $a.PSTypeNames[0] -eq "IntelEthernetAdapter")
                {
                    $AdapterNames += $a.Name
                }
                else
                {
                    $ErrorMessages.Value += $Messages.InvalidObject -f $a.GetType().Name
                }
            }
        }
        else
        {
            $ErrorMessages.Value += $Messages.InvalidParamsAdapterOrName
            break
        }
    } while ($false)

    return $AdapterNames
}

Function ValidatePathParams([ref]$LogPath, $UseDefaultPath, $LogName, [ref]$ErrorMessages)
{
    $Result = $true

    try
    {
        if ($UseDefaultPath)
        {
            $DefaultPath = $ENV:LOCALAPPDATA + "\Intel\Wired Networking\" + $LogName

            if (-not (Test-Path -Path $DefaultPath -ErrorAction Stop))
            {
                New-Item -Path $ENV:LOCALAPPDATA -Name "\Intel\Wired Networking" -ItemType "directory" -ErrorAction SilentlyContinue
            }
            $LogPath.Value = $DefaultPath
        }
        else
        {
            $LogPath.Value = $Path
        }

        $isPathFile = Test-Path -Path $LogPath.Value -PathType Leaf -ErrorAction Stop

        if (($isPathFile) -and (-not $Append) -and (-not $Force))
        {
            $ErrorMessages.Value += $Messages.LogmanFileExists -f $AdapterName
            $Result = $false
        }
        elseif (-not $isPathFile)
        {
            if (Test-Path -Path $LogPath.Value -ErrorAction Stop)
            {
                $ErrorMessages.Value += $Messages.FolderFileNameExits
                $Result = $false
            }
            else
            {
                $AbsolutePath = [IO.Path]::GetFullPath($LogPath.Value)
                $ParentFolder = Split-Path -Path $AbsolutePath

                if (-Not (Test-Path -Path $ParentFolder -ErrorAction Stop))
                {
                    $ErrorMessages.Value += $Messages.PathIncorrect
                    $Result = $false
                }
            }
        }
    }
    catch
    {
        $ErrorMessages.Value += $Messages.PathIncorrect
        $Result = $false
    }
    return $Result
}


function InvokeCimMethod($ClassName, $InstanceName = "", $MethodName, $params = @{}, $Namespace = "root\wmi")
{
    $query = "Select * from $ClassName"
    if ($InstanceName)
    {
        $query += " where instancename like '$InstanceName'"
    }

    Invoke-CimMethod -Query $query -MethodName $MethodName -Arguments $params -Namespace $Namespace -ErrorAction SilentlyContinue
}

function GetIntelEthernetDevices()
{
    $script:PnpDevice = Get-PnpDevice | Where-Object { $_.Class -eq "Net" } -ErrorAction SilentlyContinue
    if ($script:PnpDevice)
    {
        $script:SupportedAdapters = $script:PnpDevice.Where({ $_.Service -in @('icea', 'scea', 'i40ea', 'i40eb') })
    }
}

function GetAllSupportedEthernetDevices()
{
    $script:PnpDevice = Get-PnpDevice | Where-Object { $_.Class -eq "Net" } -ErrorAction SilentlyContinue
    if ($script:PnpDevice)
    {
        $DriverWhereScriptBlock = { $_.Service -in @('icea', 'scea', 'i40ea', 'i40eb') -or $_.Service -like 'icei*' -or $_.Service -like 'i40ei*' }
        $script:SupportedAdapters = $script:PnpDevice.Where($DriverWhereScriptBlock)
    }
}

function GetSupportedAdapters($AdapterNames, [ref]$WarningMessages)
{
    $SupportedAdapterNames = @()
    $AdapterNames = $AdapterNames | Sort-Object
    foreach ($a in $AdapterNames)
    {
        if ($script:SupportedAdapters.Name -Contains $a)
        {
            $SupportedAdapterNames += $a
        }
        else
        {
            $WarningMessages.Value += $Messages.NoCmdletSupport -f $a
        }
    }

    return $SupportedAdapterNames
}

function CheckDeviceError($AdapterName)
{
    $SupportedAdapter = $script:SupportedAdapters.Where({ $_.FriendlyName -eq $AdapterName })

    if ($SupportedAdapter)
    {
        # if the device is not 'working properly'
        if ([Int32]$SupportedAdapter.ConfigManagerErrorCode -ne 0)
        {
            $PreErrorActionPreference = $global:ErrorActionPreference
            $global:ErrorActionPreference = 'SilentlyContinue'
            # Workaround due to ProblemDescription being empty by default - change current path
            Push-Location -Path (Get-Module -Name PnPDevice).ModuleBase
            $StatusMsg = $AdapterName + ": " + $SupportedAdapter.ProblemDescription
            # Reset path
            Pop-Location
            $global:ErrorActionPreference = $PreErrorActionPreference
            $StatusMsg
        }
    }
}

function ValidateSingleAdapter([array]$PipelineInput, [array]$AdapterName, [ref]$ErrorMessages)
{
    $Result = $false

    do
    {
        if ($PipelineInput.Count -gt 1)
        {
            $ErrorMessages.Value += $Messages.InvalidParams
            break
        }

        if ($AdapterName.Count -gt 1)
        {
            $ErrorMessages.Value += $Messages.InvalidParams
            break
        }

        $Result = $true
    } while ($false)

    return $Result
}

function IsWin2022ServerOrLater()
{
    $WIN2022_BUILD = 20298
    $OSVersion = [Environment]::OSVersion.Version
    $iMajorVersion = $OSVersion.Major
    $iBuildNumber = $OSVersion.Build

    if (($iMajorVersion -eq 10) -and ($iBuildNumber -ge $WIN2022_BUILD) -and -not (IsOperatingSystemClientBased))
    {
        return $true
    }

    return $false
}

function IsOperatingSystemClientBased()
{
    $VER_NT_WORKSTATION = 1
    return ($VER_NT_WORKSTATION -eq ((Get-CimInstance Win32_OperatingSystem -ErrorAction SilentlyContinue).ProductType))
}

function CheckPropertyExists($TestObject, $strPropertyName)
{
    $bReturnValue = $true
    try
    {
        return (($TestObject | Get-Member -MemberType "Property" -ErrorAction Stop).Name -contains $strPropertyName)
    }
    catch
    {
        $bReturnValue = $false
    }
    return $bReturnValue
}

# SIG # Begin signature block
# MIIorAYJKoZIhvcNAQcCoIIonTCCKJkCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBbPR7uqXIQGjLA
# 8C2cFM9c6FPQ6MI7PpBqnbjremPzlKCCEgUwggWeMIIEhqADAgECAhEAzS1l4rws
# CIvYBjRVawV4ujANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQGEwJHQjEbMBkGA1UE
# CBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYDVQQHEwdTYWxmb3JkMRgwFgYDVQQK
# Ew9TZWN0aWdvIExpbWl0ZWQxJDAiBgNVBAMTG1NlY3RpZ28gUlNBIENvZGUgU2ln
# bmluZyBDQTAeFw0yMTA0MDIwMDAwMDBaFw0yMzA0MDIyMzU5NTlaMIGEMQswCQYD
# VQQGEwJVUzETMBEGA1UECAwKQ2FsaWZvcm5pYTEUMBIGA1UEBwwLU2FudGEgQ2xh
# cmExGjAYBgNVBAoMEUludGVsIENvcnBvcmF0aW9uMRIwEAYDVQQLDAlTb2xhcktN
# Q1MxGjAYBgNVBAMMEUludGVsIENvcnBvcmF0aW9uMIIBojANBgkqhkiG9w0BAQEF
# AAOCAY8AMIIBigKCAYEA7CCN9iKpDHOrRceKhlXFP/tf6Lllw2H2fR9KVI4/fQIx
# MU1hXwnlHmAzMCY7IgcCFY4p3F5/MJGKaqYngwOo28Zo6Q1N6ukysA7PSavmF2RY
# WD6VFeya/2H0PoNeRFjHaRzSeynFFeJAFew9r7UReUwM/507sxZYPQuWWIdAEK7H
# Dqp2VlHmgZOXVGHhNO6GFOKpC/C01g6X3x6OquddRNMt5UrZzZzDo5MpJz9SBB2V
# jiqwZ80dvNR2W2xi90cIHh4BkXvB54UNkp4VTVu16T0k3cweo+C39U7GrCAr5Axz
# DETjBvhNtP1sf9SoRV7xY6g5wssfI7yYT9J0gsifn/Vy8MWH355TPoA+PVhbAu0m
# 9FMz4EWu55nnUurNML2jaUxsos21/7ELat12kWC0tq9fhkODjKO8X9PuiBHflZLk
# d3F4QcSMvuGocWGqE77VV3vn8jlvigm2TOV0CfGTQajGMX0jeTRZ19fzBNkt2X9d
# SSGolI/Kj1gSvCggpkUBAgMBAAGjggGQMIIBjDAfBgNVHSMEGDAWgBQO4TqoUzox
# 1Yq+wbutZxoDha00DjAdBgNVHQ4EFgQUshkNuM2SdwJnW4vFy8c4FtUTrbQwDgYD
# VR0PAQH/BAQDAgeAMAwGA1UdEwEB/wQCMAAwEwYDVR0lBAwwCgYIKwYBBQUHAwMw
# EQYJYIZIAYb4QgEBBAQDAgQQMEoGA1UdIARDMEEwNQYMKwYBBAGyMQECAQMCMCUw
# IwYIKwYBBQUHAgEWF2h0dHBzOi8vc2VjdGlnby5jb20vQ1BTMAgGBmeBDAEEATBD
# BgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29S
# U0FDb2RlU2lnbmluZ0NBLmNybDBzBggrBgEFBQcBAQRnMGUwPgYIKwYBBQUHMAKG
# Mmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1JTQUNvZGVTaWduaW5nQ0Eu
# Y3J0MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG
# 9w0BAQsFAAOCAQEAVadLNRW4f/pKMqrbn0BdOoQ8/1EJ87gvVfosei2bLwTEvpmv
# mn2n561H6AFedtIJ6L4FmXII4M4r20i+5LREbI6PpKDmOAf4xW7POxfCRvkTQAZO
# 3zoVxjMQBXo7cZVF1xHCdviXzD1usuIiCF8DLm6z4O/kyeFFNcn816yPQct91Pnk
# SBBVvL+Kwu8xvR+ZIQy632WUA4HnNpRdFnVSzUifEg2GrtsKZR8k+rm2o8K8yjJq
# 3SznwgJQCMVMh3CtRtUwE/c7o/6rvm53fTYJDd3aoPHVgH6S2WqS3+3mQG7A6hTD
# nrP/mYnS4PF7XzxxjZhUlhy4G/MarJPvT9IrNDCCBfUwggPdoAMCAQICEB2iSDBv
# myYY0ILgln0z02owDQYJKoZIhvcNAQEMBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0eTEeMBwGA1UEChMV
# VGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1c3QgUlNBIENl
# cnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTE4MTEwMjAwMDAwMFoXDTMwMTIzMTIz
# NTk1OVowfDELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFuY2hlc3Rl
# cjEQMA4GA1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSQw
# IgYDVQQDExtTZWN0aWdvIFJTQSBDb2RlIFNpZ25pbmcgQ0EwggEiMA0GCSqGSIb3
# DQEBAQUAA4IBDwAwggEKAoIBAQCGIo0yhXoYn0nwli9jCB4t3HyfFM/jJrYlZilA
# hlRGdDFixRDtsocnppnLlTDAVvWkdcapDlBipVGREGrgS2Ku/fD4GKyn/+4uMyD6
# DBmJqGx7rQDDYaHcaWVtH24nlteXUYam9CflfGqLlR5bYNV+1xaSnAAvaPeX7Wpy
# vjg7Y96Pv25MQV0SIAhZ6DnNj9LWzwa0VwW2TqE+V2sfmLzEYtYbC43HZhtKn52B
# xHJAteJf7wtF/6POF6YtVbC3sLxUap28jVZTxvC6eVBJLPcDuf4vZTXyIuosB69G
# 2flGHNyMfHEo8/6nxhTdVZFuihEN3wYklX0Pp6F8OtqGNWHTAgMBAAGjggFkMIIB
# YDAfBgNVHSMEGDAWgBRTeb9aqitKz1SA4dibwJ3ysgNmyzAdBgNVHQ4EFgQUDuE6
# qFM6MdWKvsG7rWcaA4WtNA4wDgYDVR0PAQH/BAQDAgGGMBIGA1UdEwEB/wQIMAYB
# Af8CAQAwHQYDVR0lBBYwFAYIKwYBBQUHAwMGCCsGAQUFBwMIMBEGA1UdIAQKMAgw
# BgYEVR0gADBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8vY3JsLnVzZXJ0cnVzdC5j
# b20vVVNFUlRydXN0UlNBQ2VydGlmaWNhdGlvbkF1dGhvcml0eS5jcmwwdgYIKwYB
# BQUHAQEEajBoMD8GCCsGAQUFBzAChjNodHRwOi8vY3J0LnVzZXJ0cnVzdC5jb20v
# VVNFUlRydXN0UlNBQWRkVHJ1c3RDQS5jcnQwJQYIKwYBBQUHMAGGGWh0dHA6Ly9v
# Y3NwLnVzZXJ0cnVzdC5jb20wDQYJKoZIhvcNAQEMBQADggIBAE1jUO1HNEphpNve
# aiqMm/EAAB4dYns61zLC9rPgY7P7YQCImhttEAcET7646ol4IusPRuzzRl5ARokS
# 9At3WpwqQTr81vTr5/cVlTPDoYMot94v5JT3hTODLUpASL+awk9KsY8k9LOBN9O3
# ZLCmI2pZaFJCX/8E6+F0ZXkI9amT3mtxQJmWunjxucjiwwgWsatjWsgVgG10Xkp1
# fqW4w2y1z99KeYdcx0BNYzX2MNPPtQoOCwR/oEuuu6Ol0IQAkz5TXTSlADVpbL6f
# ICUQDRn7UJBhvjmPeo5N9p8OHv4HURJmgyYZSJXOSsnBf/M6BZv5b9+If8AjntIe
# Q3pFMcGcTanwWbJZGehqjSkEAnd8S0vNcL46slVaeD68u28DECV3FTSK+TbMQ5Lk
# uk/xYpMoJVcp+1EZx6ElQGqEV8aynbG8HArafGd+fS7pKEwYfsR7MUFxmksp7As9
# V1DSyt39ngVR5UR43QHesXWYDVQk/fBO4+L4g71yuss9Ou7wXheSaG3IYfmm8SoK
# C6W59J7umDIFhZ7r+YMp08Ysfb06dy6LN0KgaoLtO0qqlBCk4Q34F8W2WnkzGJLj
# tXX4oemOCiUe5B7xn1qHI/+fpFGe+zmAEc3btcSnqIBv5VPU4OOiwtJbGvoyJi1q
# V3AcPKRYLqPzW0sH3DJZ84enGm1YMIIGZjCCBE6gAwIBAgITMwAAAES3P/zvWs+i
# egAAAAAARDANBgkqhkiG9w0BAQUFADB/MQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMSkwJwYDVQQDEyBNaWNyb3NvZnQgQ29kZSBWZXJpZmljYXRp
# b24gUm9vdDAeFw0xNTA3MjIyMTAzNDlaFw0yNTA3MjIyMTAzNDlaMIGIMQswCQYD
# VQQGEwJVUzETMBEGA1UECBMKTmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENp
# dHkxHjAcBgNVBAoTFVRoZSBVU0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNF
# UlRydXN0IFJTQSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eTCCAiIwDQYJKoZIhvcN
# AQEBBQADggIPADCCAgoCggIBAIASZRc2DsPbCLPQrFcNdu3NJ9NMrVCDYeKqIE0J
# LWQJ3M6Jn8w9qez2z8Hc8dOx1ns3KBErR9o5xrw6GbRfpr19naNjQrZ28qk7K5H4
# 4m/Q7BYgkAk+4uh0yRi0kdRiZNt/owbxiBhqkCI8vP4T8IcUe/bkH47U5FHGEWdG
# CFHLhhRUP7wz/n5snP8WnRi9UY41pqdmyHJn2yFmsdSbeAPAUDrozPDcvJ5M/q8F
# ljUfV1q3/875PbcstvZU3cjnEjpNrkyKt1yatLcgPcp/IjSufjtoZgFE5wFORlOb
# M2D3lL5TN5BzQ/Myw1Pv26r+dE5px2uMYJPexMcM3+EyrsyTO1F4lWeL7j1W/gzQ
# aQ8bD/MlJmszbfduR/pzQ+V+DqVmsSl8MoRjVYnEDcGTVDAZE6zTfTen6106bDVc
# 20HXEtqpSQvf2ICKCZNijrVmzyWIzYS4sT+kOQ/ZAp7rEkyVfPNrBaleFoPMuGfi
# 6BOdzFuC00yz7Vv/3uVzrCM7LQC/NVV0CUnYSVgaf5I25lGSDvMmfRxNF7zJ7EMm
# 0L9BX0CpRET0medXh55QH1dUqD79dGMvsVBlCeZYQi5DGky08CVHWfoEHpPUJkZK
# UIGy3r54t/xnFeHJV4QeD2PW6WK61l9VLupcxigIBCU5uA4rqfJMlxwHPw1S9e3v
# L4IPAgMBAAGjgdAwgc0wEwYDVR0lBAwwCgYIKwYBBQUHAwMwEgYDVR0TAQH/BAgw
# BgEB/wIBAjAdBgNVHQ4EFgQUU3m/WqorSs9UgOHYm8Cd8rIDZsswCwYDVR0PBAQD
# AgGGMB8GA1UdIwQYMBaAFGL7CiFbf0NuEdoJVFBr9dKWcfGeMFUGA1UdHwROMEww
# SqBIoEaGRGh0dHA6Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3Rz
# L01pY3Jvc29mdENvZGVWZXJpZlJvb3QuY3JsMA0GCSqGSIb3DQEBBQUAA4ICAQBr
# IpM8PTlUcWRrDvLkPDARxSBKS4YPkvH/M3k62eSYpw5AoCKAfmGy4KcZzyaVMSpl
# 1GpPMYbqwMYuxWSMPUhZzQsvdD2UJhMQQtSXmCdePHbSeGkdGmTnBXJ14OtmQEOf
# jwxG/5dgpshnrRAIm2Km6b46itMHTZ9ykyW8BhHgLJA4Pmcc/RnXnpDOPcLg52Gs
# wOUE9R6ZVAyRDQFWcTeuJ9SeQyKlySfNTeVxEjkkpUFWh/+8VRQPJcqJ7seX5dIT
# /z1+GqCPP8gs16Nw0MdgwPzYPlHnl8Y+O+3PeL6KyuPE8qen7Z6uCAKPoFLbch7V
# O8NNn476m3DH+OO/bD+Sm+Q3PuxqjCn5waK/iz4aaWb7HGNPJgHJAsQ+0v/DQ6gb
# /Zn61LylueKTLzsBxdH0Oi9ow+Bkt1qVXkbMB4NpuzwFklZzNXNFmE582BKlt0Lp
# omP2QmAYcNE7bzHAh8fmceHzRhbp9bhys+ltH2ImSaNJi91ox4toVvfe/PqHJLgD
# gReP5fFnah2u03T3jKVdswuOQimWzknEd35mfAEXGmwUJMOwF3cF2BpAt4Zr2OR7
# QKx+305vJPkggIKMM+fl+inYndqLcF0ryR2CTAtny4RBnucGfhGDRC2KGe70f5rd
# eRw3GR6fP4wpug1cEIY3bEjNRV3NcLy80U1d2MW4djGCFf0wghX5AgEBMIGRMHwx
# CzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNV
# BAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEkMCIGA1UEAxMb
# U2VjdGlnbyBSU0EgQ29kZSBTaWduaW5nIENBAhEAzS1l4rwsCIvYBjRVawV4ujAN
# BglghkgBZQMEAgEFAKBqMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisG
# AQQBgjcCAQsxDjAMBgorBgEEAYI3AgEWMC8GCSqGSIb3DQEJBDEiBCAm9+S0DJsZ
# 1rfd2L59p1CtOwijMC04gWFLQqhLwMfhZzANBgkqhkiG9w0BAQEFAASCAYCvMzb9
# XKGcIcyT4Xui0vdPtALte3lVJCqxIee6j+ArbGcgYGz2xKx97raEY2rVQRZkDuIv
# sN42Weo2xgcosiNn+9xuilFRvx1m+Ukju9tup1FnTNVXcv+UQP8N97t/WhFx/UTb
# vNGJvUew9dl/TXRaisNFEjPedhBGfIumvSU9QVXu6exqyuIgBFnhYZUweJ+IZMMf
# HTF5XTUvUQ0f0oYOyc4ZZFAdN+vb+tRymq1sKxQTXt6Wt60qoxDcO52vmrLvxLNF
# FQ7+LeRD2l2w4std3COqDBjqvausHePNtYXbVqDhXcEjBnmMkWxgfJM4HPOfSwj4
# S+sc9wkhaQUVCo6pHhqGtAyhFr44BZT+lz6xKnV46iFs7biuuYCNi97SKCLE3exj
# LBc/ao+LYKAEYTYDwaNctt0slzA/5rf7qJJcvkY5nW91QQvrYxdp/zVYt/SM3tye
# rHIfqj2oVr3/qvWMoc7k0kdtOO5kvTK/mqO3MY7C3KzEnvH3YojYIVjIvwOhghNQ
# MIITTAYKKwYBBAGCNwMDATGCEzwwghM4BgkqhkiG9w0BBwKgghMpMIITJQIBAzEP
# MA0GCWCGSAFlAwQCAgUAMIHvBgsqhkiG9w0BCRABBKCB3wSB3DCB2QIBAQYKKwYB
# BAGyMQIBATAxMA0GCWCGSAFlAwQCAQUABCBDoc8k4s2WAqs2TvHaZ6fk6hyPGhiz
# P2UC0ZuYvYVKTwIUI53n3rqXt1BaFg6BkDC1CpbdoegYDzIwMjMwMTEwMjMwMDAx
# WqBupGwwajELMAkGA1UEBhMCR0IxEzARBgNVBAgTCk1hbmNoZXN0ZXIxGDAWBgNV
# BAoTD1NlY3RpZ28gTGltaXRlZDEsMCoGA1UEAwwjU2VjdGlnbyBSU0EgVGltZSBT
# dGFtcGluZyBTaWduZXIgIzOggg3qMIIG9jCCBN6gAwIBAgIRAJA5f5rSSjoT8r2R
# Xwg4qUMwDQYJKoZIhvcNAQEMBQAwfTELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdy
# ZWF0ZXIgTWFuY2hlc3RlcjEQMA4GA1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2Vj
# dGlnbyBMaW1pdGVkMSUwIwYDVQQDExxTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5n
# IENBMB4XDTIyMDUxMTAwMDAwMFoXDTMzMDgxMDIzNTk1OVowajELMAkGA1UEBhMC
# R0IxEzARBgNVBAgTCk1hbmNoZXN0ZXIxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRl
# ZDEsMCoGA1UEAwwjU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBTaWduZXIgIzMw
# ggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQCQsnE/eeHUuYoXzMOXwpCU
# cu1aOm8BQ39zWiifJHygNUAG+pSvCqGDthPkSxUGXmqKIDRxe7slrT9bCqQfL2x9
# LmFR0IxZNz6mXfEeXYC22B9g480Saogfxv4Yy5NDVnrHzgPWAGQoViKxSxnS8JbJ
# RB85XZywlu1aSY1+cuRDa3/JoD9sSq3VAE+9CriDxb2YLAd2AXBF3sPwQmnq/ybM
# A0QfFijhanS2nEX6tjrOlNEfvYxlqv38wzzoDZw4ZtX8fR6bWYyRWkJXVVAWDUt0
# cu6gKjH8JgI0+WQbWf3jOtTouEEpdAE/DeATdysRPPs9zdDn4ZdbVfcqA23VzWLa
# zpwe/OpwfeZ9S2jOWilh06BcJbOlJ2ijWP31LWvKX2THaygM2qx4Qd6S7w/F7Kvf
# LW8aVFFsM7ONWWDn3+gXIqN5QWLP/Hvzktqu4DxPD1rMbt8fvCKvtzgQmjSnC//+
# HV6k8+4WOCs/rHaUQZ1kHfqA/QDh/vg61MNeu2lNcpnl8TItUfphrU3qJo5t/KlI
# mD7yRg1psbdu9AXbQQXGGMBQ5Pit/qxjYUeRvEa1RlNsxfThhieThDlsdeAdDHpZ
# iy7L9GQsQkf0VFiFN+XHaafSJYuWv8at4L2xN/cf30J7qusc6es9Wt340pDVSZo6
# HYMaV38cAcLOHH3M+5YVxQIDAQABo4IBgjCCAX4wHwYDVR0jBBgwFoAUGqH4YRkg
# D8NBd0UojtE1XwYSBFUwHQYDVR0OBBYEFCUuaDxrmiskFKkfot8mOs8UpvHgMA4G
# A1UdDwEB/wQEAwIGwDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUF
# BwMIMEoGA1UdIARDMEEwNQYMKwYBBAGyMQECAQMIMCUwIwYIKwYBBQUHAgEWF2h0
# dHBzOi8vc2VjdGlnby5jb20vQ1BTMAgGBmeBDAEEAjBEBgNVHR8EPTA7MDmgN6A1
# hjNodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29SU0FUaW1lU3RhbXBpbmdD
# QS5jcmwwdAYIKwYBBQUHAQEEaDBmMD8GCCsGAQUFBzAChjNodHRwOi8vY3J0LnNl
# Y3RpZ28uY29tL1NlY3RpZ29SU0FUaW1lU3RhbXBpbmdDQS5jcnQwIwYIKwYBBQUH
# MAGGF2h0dHA6Ly9vY3NwLnNlY3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBz
# 2u1ocsvCuUChMbu0A6MtFHsk57RbFX2o6f2t0ZINfD02oGnZ85ow2qxp1nRXJD9+
# DzzZ9cN5JWwm6I1ok87xd4k5f6gEBdo0wxTqnwhUq//EfpZsK9OU67Rs4EVNLLL3
# OztatcH714l1bZhycvb3Byjz07LQ6xm+FSx4781FoADk+AR2u1fFkL53VJB0ngtP
# TcSqE4+XrwE1K8ubEXjp8vmJBDxO44ISYuu0RAx1QcIPNLiIncgi8RNq2xgvbnit
# xAW06IQIkwf5fYP+aJg05Hflsc6MlGzbA20oBUd+my7wZPvbpAMxEHwa+zwZgNEL
# cLlVX0e+OWTOt9ojVDLjRrIy2NIphskVXYCVrwL7tNEunTh8NeAPHO0bR0icImpV
# gtnyughlA+XxKfNIigkBTKZ58qK2GpmU65co4b59G6F87VaApvQiM5DkhFP8KvrA
# p5eo6rWNes7k4EuhM6sLdqDVaRa3jma/X/ofxKh/p6FIFJENgvy9TZntyeZsNv53
# Q5m4aS18YS/to7BJ/lu+aSSR/5P8V2mSS9kFP22GctOi0MBk0jpCwRoD+9DtmiG4
# P6+mslFU1UzFyh8SjVfGOe1c/+yfJnatZGZn6Kow4NKtt32xakEnbgOKo3TgigmC
# br/j9re8ngspGGiBoZw/bhZZSxQJCZrmrr9gFd2G9TCCBuwwggTUoAMCAQICEDAP
# b6zdZph0fKlGNqd4LbkwDQYJKoZIhvcNAQEMBQAwgYgxCzAJBgNVBAYTAlVTMRMw
# EQYDVQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0eTEeMBwGA1UE
# ChMVVGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1c3QgUlNB
# IENlcnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTE5MDUwMjAwMDAwMFoXDTM4MDEx
# ODIzNTk1OVowfTELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFuY2hl
# c3RlcjEQMA4GA1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVk
# MSUwIwYDVQQDExxTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5nIENBMIICIjANBgkq
# hkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAyBsBr9ksfoiZfQGYPyCQvZyAIVSTuc+g
# PlPvs1rAdtYaBKXOR4O168TMSTTL80VlufmnZBYmCfvVMlJ5LsljwhObtoY/AQWS
# Zm8hq9VxEHmH9EYqzcRaydvXXUlNclYP3MnjU5g6Kh78zlhJ07/zObu5pCNCrNAV
# w3+eolzXOPEWsnDTo8Tfs8VyrC4Kd/wNlFK3/B+VcyQ9ASi8Dw1Ps5EBjm6dJ3VV
# 0Rc7NCF7lwGUr3+Az9ERCleEyX9W4L1GnIK+lJ2/tCCwYH64TfUNP9vQ6oWMilZx
# 0S2UTMiMPNMUopy9Jv/TUyDHYGmbWApU9AXn/TGs+ciFF8e4KRmkKS9G493bkV+f
# PzY+DjBnK0a3Na+WvtpMYMyou58NFNQYxDCYdIIhz2JWtSFzEh79qsoIWId3pBXr
# GVX/0DlULSbuRRo6b83XhPDX8CjFT2SDAtT74t7xvAIo9G3aJ4oG0paH3uhrDvBb
# fel2aZMgHEqXLHcZK5OVmJyXnuuOwXhWxkQl3wYSmgYtnwNe/YOiU2fKsfqNoWTJ
# iJJZy6hGwMnypv99V9sSdvqKQSTUG/xypRSi1K1DHKRJi0E5FAMeKfobpSKupcNN
# gtCN2mu32/cYQFdz8HGj+0p9RTbB942C+rnJDVOAffq2OVgy728YUInXT50zvRq1
# naHelUF6p4MCAwEAAaOCAVowggFWMB8GA1UdIwQYMBaAFFN5v1qqK0rPVIDh2JvA
# nfKyA2bLMB0GA1UdDgQWBBQaofhhGSAPw0F3RSiO0TVfBhIEVTAOBgNVHQ8BAf8E
# BAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIBADATBgNVHSUEDDAKBggrBgEFBQcDCDAR
# BgNVHSAECjAIMAYGBFUdIAAwUAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC51
# c2VydHJ1c3QuY29tL1VTRVJUcnVzdFJTQUNlcnRpZmljYXRpb25BdXRob3JpdHku
# Y3JsMHYGCCsGAQUFBwEBBGowaDA/BggrBgEFBQcwAoYzaHR0cDovL2NydC51c2Vy
# dHJ1c3QuY29tL1VTRVJUcnVzdFJTQUFkZFRydXN0Q0EuY3J0MCUGCCsGAQUFBzAB
# hhlodHRwOi8vb2NzcC51c2VydHJ1c3QuY29tMA0GCSqGSIb3DQEBDAUAA4ICAQBt
# VIGlM10W4bVTgZF13wN6MgstJYQRsrDbKn0qBfW8Oyf0WqC5SVmQKWxhy7VQ2+J9
# +Z8A70DDrdPi5Fb5WEHP8ULlEH3/sHQfj8ZcCfkzXuqgHCZYXPO0EQ/V1cPivNVY
# eL9IduFEZ22PsEMQD43k+ThivxMBxYWjTMXMslMwlaTW9JZWCLjNXH8Blr5yUmo7
# Qjd8Fng5k5OUm7Hcsm1BbWfNyW+QPX9FcsEbI9bCVYRm5LPFZgb289ZLXq2jK0KK
# IZL+qG9aJXBigXNjXqC72NzXStM9r4MGOBIdJIct5PwC1j53BLwENrXnd8ucLo0j
# GLmjwkcd8F3WoXNXBWiap8k3ZR2+6rzYQoNDBaWLpgn/0aGUpk6qPQn1BWy30mRa
# 2Coiwkud8TleTN5IPZs0lpoJX47997FSkc4/ifYcobWpdR9xv1tDXWU9UIFuq/DQ
# 0/yysx+2mZYm9Dx5i1xkzM3uJ5rloMAMcofBbk1a0x7q8ETmMm8c6xdOlMN4ZSA7
# D0GqH+mhQZ3+sbigZSo04N6o+TzmwTC7wKBjLPxcFgCo0MR/6hGdHgbGpm0yXbQ4
# CStJB6r97DDa8acvz7f9+tCjhNknnvsBZne5VhDhIG7GrrH5trrINV0zdo7xfCAM
# KneutaIChrop7rRaALGMq+P5CslUXdS5anSevUiumDGCBC0wggQpAgEBMIGSMH0x
# CzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNV
# BAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDElMCMGA1UEAxMc
# U2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBDQQIRAJA5f5rSSjoT8r2RXwg4qUMw
# DQYJYIZIAWUDBAICBQCgggFrMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0BCRABBDAc
# BgkqhkiG9w0BCQUxDxcNMjMwMTEwMjMwMDAxWjA/BgkqhkiG9w0BCQQxMgQwb219
# syT/MJN7OE0RXSqBo3RhhQSWJXY2IOEk5QGdG7uo73MxK9KLgcKr+MXpcK9VMIHt
# BgsqhkiG9w0BCRACDDGB3TCB2jCB1zAWBBSrNAE6rECXMZ8IGvCzGOGD+A94gTCB
# vAQUAtZbleKDcMFXAJX6iPkj3ZN/rY8wgaMwgY6kgYswgYgxCzAJBgNVBAYTAlVT
# MRMwEQYDVQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0eTEeMBwG
# A1UEChMVVGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1c3Qg
# UlNBIENlcnRpZmljYXRpb24gQXV0aG9yaXR5AhAwD2+s3WaYdHypRjaneC25MA0G
# CSqGSIb3DQEBAQUABIICAEKMJ3stQ+coQIMO8cWK4HxR53cyIKYhvhEuJWXV1gyz
# wiW8t05edOBlqkIYRn0oK/d8ZUTEJZm5rDQDjLwXzGzjSNAb1fXo0c0ylUXUZKII
# yxrzDmeestAzV6dTYZ3K5c69CRrmZFLxyf+qIR+VxTGKxEwLibH2e+5QCA4gQtYe
# 8As+G96n1I+BPZuCbkPxLNFaZHe6DsZpmt3BojXtUhCG4PLHLhP3SVWwAG5LdRg4
# b40Y23WBTY+jt4whPE/eRJ/v4x2dIHjKhjv2JeR62yk0GtMNDU1BGpkguVHkIGjz
# L3vfJKejLrGNE7rQ3lmrmALnx3UDydGihYCBvI/B0qLkrPIrMsvVmjETfmbiD9Wd
# iXix4XcKQJeXNvM7/lb0ofEwqb/8Zb66laXLRs/CfWIdOX1C0q3Zs99oXCzfUraC
# JKQByzQePhNMZRX8/8MlfQWrYRkheNtUEO8ED1q0/ibZmleHtxieHJEF8D2MtfOP
# VkvzR144RMBsOuDBOOorj2sfE4Et3HBssa0ucfGuS6IGGXzwYXk9NUKRlIxVVmny
# lOHFyeAWV7atXgpAIA/AREgGrV01IDGSOwxqchGS7WUSDqLJZe/duta/1BoseRMZ
# suuKLcxVBCo+8lUoO6+pVEjjsTRecpAMI0p/kaDW4mFJtgbxxQfpchodMARet7M2
# SIG # End signature block
