# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2021 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

# Suppress irrelevant PS Script Analyzer warnings (trailing Param() is needed to help PSSA parse the file)
[Diagnostics.CodeAnalysis.SuppressMessageAttribute('PSAvoidGlobalVars', 'Global:FormatEnumerationLimit')]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSAvoidUsingPositionalParameters", "", Scope="function")]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSUseShouldProcessForStateChangingFunctions", "", Scope="function")] Param()

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Get-IntelEthernetLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null,
    [parameter(Mandatory=$false,Position=1)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Module = '',
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Configuration = ''
    )
    Begin
    {
        $AdapterName = $Name
        $ModuleNames = $Module
        $ConfigurationNames = $Configuration
        $script:ErrorMessagesGet = @()
        $script:WarningMessagesGet = @()
        $LogConfigs = @()
        $DefaultFormatEnumLimit = $global:FormatEnumerationLimit
        $global:FormatEnumerationLimit = -1
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue
        }
    }
    Process
    {
        $Adapters = $Adapter

        $AdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapters ([ref]$script:ErrorMessagesGet)

        foreach ($a in $AdapterNames)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:WarningMessagesGet += $TmpStatusMsg
                continue
            }

            $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiGetFwLogLevels"
            if (!(ValidateMethodOutput $MethodOutput "Get"))
            {
                continue
            }

            $Local:IntelLogConfigurations = @{}
            $IntelLogConfigurations = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIGURATIONS" ([ref]$script:WarningMessagesGet)
            $Local:IntelLogConfigModules = @{}
            $IntelLogConfigModules = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIG_MODULES" ([ref]$script:WarningMessagesGet)
            $Local:IntelLogConfigLevels = @{}
            $IntelLogConfigLevels = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIG_LEVELS" ([ref]$script:WarningMessagesGet)
            if ($IntelLogConfigurations.Count -eq 0 -or $IntelLogConfigModules.Count -eq 0 -or $IntelLogConfigLevels.Count -eq 0)
            {
                continue
            }

            $DisplayConfigNames = @()
            if (!(ValidateConfigurationParam $ConfigurationNames $ModuleNames ([ref]$DisplayConfigNames) $a))
            {
                break
            }

            $DisplayModules = @()
            $DisplayLevels = @()

            if ($DisplayConfigNames)
            {
                GetConfigLogs $DisplayConfigNames ([ref]$DisplayModules) ([ref]$DisplayLevels) $MethodOutput
            }
            else
            {
                ValidateGetModuleParam ([ref]$DisplayModules) ([ref]$DisplayLevels) $ModuleNames $MethodOutput $a
            }

            if ($DisplayModules -and $DisplayLevels)
            {
                $LogConfigs += GatherOutput $a $DisplayModules $DisplayLevels
            }
        }
    }
    End
    {
        $global:FormatEnumerationLimit = $DefaultFormatEnumLimit
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }

        foreach ($WarningMessage in $script:WarningMessagesGet)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesGet)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Set-IntelEthernetLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null,
    [parameter(Mandatory=$false,Position=1)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Module = '',
    [parameter(Mandatory=$true,Position=2)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String]
    $Level = '',
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Configuration = ''
    )
    Begin
    {
        $ModuleNames = $Module
        $LevelName = $Level
        $ConfigurationNames = $Configuration
        $script:WarningMessagesSet = @()
        $script:ErrorMessagesSet = @()
        $LogConfigs = @()
        GetIntelEthernetDevices
        # this cmdlet only supports these two drivers
        $script:SupportedAdapters = $script:SupportedAdapters.Where({$_.Service -in @('icea', 'scea')})
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue
        }
    }
    Process
    {
        do
        {
            $PreProcessedAdapterNames = ValidateSetAdapterNameParams $Name $Adapter ([ref]$script:ErrorMessagesSet)
            $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:ErrorMessagesSet))

            if ([string]::IsNullOrEmpty($ConfigurationNames) -and ([string]::IsNullOrEmpty($ModuleNames)))
            {
                $script:ErrorMessagesSet += $Messages.InvalidParamsConfigOrModule
                break
            }

            foreach($a in $AdapterNames)
            {
                $TmpStatusMsg = CheckDeviceError $a
                if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
                {
                    $script:ErrorMessagesSet += $TmpStatusMsg
                    continue
                }

                $DisplayModules = @()
                $DisplayLevels = @()
                $InputModulesId = [uint32[]]::new(64)
                $InputLevelId = [uint32[]]::new(64)

                $Local:IntelLogConfigurations = @{}
                $IntelLogConfigurations = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIGURATIONS" ([ref]$script:WarningMessagesSet)
                $Local:IntelLogConfigModules = @{}
                $IntelLogConfigModules = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIG_MODULES" ([ref]$script:WarningMessagesSet)
                $Local:IntelLogConfigLevels = @{}
                $IntelLogConfigLevels = GetConfigHashTable $a "\AdditionalConfiguration\INTEL_LOG_CONFIG_LEVELS" ([ref]$script:WarningMessagesSet)
                if ($IntelLogConfigurations.Count -eq 0 -or $IntelLogConfigModules.Count -eq 0 -or $IntelLogConfigLevels.Count -eq 0)
                {
                    continue
                }

                $ModuleCount = 0
                $DisplayConfigNames = @()
                if (!(ValidateConfigurationParam $ConfigurationNames $ModuleNames ([ref]$DisplayConfigNames) $a))
                {
                    break
                }

                if ($DisplayConfigNames)
                {
                    $InputConfigurationIDs = $IntelLogConfigurations[$DisplayConfigNames]
                }
                else
                {
                    ValidateSetModuleParam ([ref]$InputModulesId) ([ref]$DisplayModules) $ModuleNames $a
                    $ModuleCount = $DisplayModules.Count
                }

                if ($ModuleCount -ne 0)
                {
                    if (!(ValidateLevelParam ([ref]$InputLevelId) ([ref]$DisplayLevels) $LevelName $ModuleCount $a))
                    {
                        break
                    }

                    $params = @{Count    = [uint32]$ModuleCount;
                                ModuleID = $InputModulesId;
                                Loglevel = $InputLevelId}

                    $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiSetFwLogLevels" $params

                    if (ValidateMethodOutput $MethodOutput "Set")
                    {
                        $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Module $DisplayModules
                    }
                }
                elseif ($null -ne $InputConfigurationIDs)
                {
                    $count = 1
                    if (!(ValidateLevelParam ([ref]$InputLevelId) ([ref]$DisplayLevels) $LevelName $count $a))
                    {
                        break
                    }

                    $bContinue = $true
                    foreach ($ConfigID in $InputConfigurationIDs)
                    {
                        $params = @{ConfigurationID = [uint32]$ConfigID;
                                    Level = $InputLevelId[0]}

                        $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiSetFwLogConfiguration" $params

                        if (!(ValidateMethodOutput $MethodOutput "Set"))
                        {
                            $bContinue = $false
                            break
                        }
                    }

                    if (!$bContinue)
                    {
                        break
                    }
                    $LogConfigs += Get-IntelEthernetLogConfig -Name $a -Configuration $DisplayConfigNames
                }
            }
        } while ($false)
    }
    End
    {
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }

        foreach ($WarningMessage in $script:WarningMessagesSet)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesSet)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Start-IntelEthernetLog
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [ValidateNotNullOrEmpty()]
    [String]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object]
    $Adapter = $null,
    [parameter(Mandatory=$false,Position=1)]
    [ValidateNotNullOrEmpty()]
    [string]
    $Path = '',
    [parameter(Mandatory=$false)]
    [switch]
    $Force,
    [parameter(Mandatory=$false)]
    [switch]
    $Append
    )
    Begin
    {
        $script:ErrorMessagesStart = @()
        $script:WarningMessagesStart = @()
        GetIntelEthernetDevices
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue
        }
    }
    End
    {
        do
        {
            if ($Name -match '\*')
            {
                $script:ErrorMessagesStart += $Messages.AdapterNotFound -f $Name
                break
            }

            $AdapterName = ValidateSetAdapterNameParams $Name $Adapter ([ref]$script:ErrorMessagesStart)

            if (([string]::IsNullOrEmpty($AdapterName)))
            {
                break
            }

            if (-not (ValidateSingleAdapter $input $AdapterName ([ref]$script:ErrorMessagesStart)))
            {
                break
            }

            $TmpStatusMsg = CheckDeviceError $AdapterName
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:ErrorMessagesStart += $TmpStatusMsg
                break
            }

            $UseDefaultPath = $false
            if (([string]::IsNullOrEmpty($Path)))
            {
                $UseDefaultPath = $true
            }

            $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $AdapterName "WmiGetFwLogLevels"

            if (ValidateMethodOutput $MethodOutput "Start")
            {
                $LogmanGUID = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "WPP_GUID" ([ref]$script:ErrorMessagesStart)
                $LogName = GetItemPropertyValue $AdapterName "\AdditionalConfiguration" "FW_LOG_NAME" ([ref]$script:ErrorMessagesStart)
                if ($null -eq $LogmanGUID -or $null -eq $LogName)
                {
                    break
                }

                if (LogmanProcessRunning $AdapterName $LogName $LogmanGUID)
                {
                    break
                }

                $LogmanPath = ''
                if (-not (ValidatePathParams ([ref]$LogmanPath) $UseDefaultPath $LogName ([ref]$script:ErrorMessagesStart)))
                {
                    break
                }

                LogmanStart $LogmanPath $LogName $LogmanGUID
                if (ValidateLogmanStart)
                {
                    $LogmanPath = Resolve-Path -Path $LogmanPath -ErrorAction SilentlyContinue
                    Write-Output $LogmanPath
                }
                break
            }
        } while ($false)

        foreach ($WarningMessage in $script:WarningMessagesStart)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesStart)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Stop-IntelEthernetLog
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [ValidateNotNullOrEmpty()]
    [String]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object]
    $Adapter = $null
    )
    Begin
    {
        $script:ErrorMessagesStop = @()
        $script:WarningMessagesStop = @()
        GetIntelEthernetDevices
        # this cmdlet only supports these two drivers
        $script:SupportedAdapters = $script:SupportedAdapters.Where({$_.Service -in @('icea', 'scea')})
        if ($script:SupportedAdapters)
        {
            $script:MSNetAdapters = Get-NetAdapter -InterfaceDescription $script:SupportedAdapters.Name -ErrorAction SilentlyContinue
        }
    }
    Process
    {
        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $Name $Adapter ([ref]$script:ErrorMessagesStop)
        $AdapterNames = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:WarningMessagesStop))

        foreach ($a in $AdapterNames)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:WarningMessagesStop += $TmpStatusMsg
                continue
            }

            $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiGetFwLogLevels"

            if (ValidateMethodOutput $MethodOutput "Stop")
            {
                $LogName = GetItemPropertyValue $a "\AdditionalConfiguration" "FW_LOG_NAME" ([ref]$script:ErrorMessagesStop)
                if ($null -eq $LogName)
                {
                    break
                }

                LogmanStop $LogName
                if (!(ValidateLogmanStop $a))
                {
                    break
                }
                break
            }
        }
    }
    End
    {
        foreach ($WarningMessage in $script:WarningMessagesStop)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesStop)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Disable-IntelEthernetLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null
    )
    Begin
    {
        $AdapterName = $Name
        $script:ErrorMessagesDisable = @()
        $script:WarningMessagesDisable = @()
        $LogConfigs = @()
        $DefaultFormatEnumLimit = $global:FormatEnumerationLimit
        $global:FormatEnumerationLimit = -1
        GetIntelEthernetDevices
    }
    Process
    {
        $Adapters = $Adapter

        $AdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapters ([ref]$script:ErrorMessagesDisable)

        foreach ($a in $AdapterNames)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:ErrorMessagesDisable += $TmpStatusMsg
                continue
            }

            $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiDisableFwLogLevels"

            if (!(ValidateMethodOutput $MethodOutput "Disable"))
            {
                continue
            }

            $LogConfigs += Get-IntelEthernetLogConfig -Name $a
        }
    }
    End
    {
        $global:FormatEnumerationLimit = $DefaultFormatEnumLimit
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }

        foreach ($WarningMessage in $script:WarningMessagesDisable)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesDisable)
        {
            Write-Error $ErrorMessage
        }
    }
}

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Reset-IntelEthernetLogConfig
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory=$false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory=$false,Position=0,ValueFromPipeline=$true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null
    )
    Begin
    {
        $AdapterName = $Name
        $script:ErrorMessagesReset = @()
        $script:WarningMessagesReset = @()
        $LogConfigs = @()
        $DefaultFormatEnumLimit = $global:FormatEnumerationLimit
        $global:FormatEnumerationLimit = -1
        GetIntelEthernetDevices
    }
    Process
    {
        $Adapters = $Adapter

        $AdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapters ([ref]$script:ErrorMessagesReset)

        foreach ($a in $AdapterNames)
        {
            $TmpStatusMsg = CheckDeviceError $a
            if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
            {
                $script:ErrorMessagesReset += $TmpStatusMsg
                continue
            }

            $MethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiResetFwLogLevels"

            if (!(ValidateMethodOutput $MethodOutput "Reset"))
            {
                continue
            }

            $LogConfigs += Get-IntelEthernetLogConfig -Name $a
        }
    }
    End
    {
        $global:FormatEnumerationLimit = $DefaultFormatEnumLimit
        if ($LogConfigs)
        {
            Write-Output $LogConfigs
        }

        foreach ($WarningMessage in $script:WarningMessagesReset)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesReset)
        {
            Write-Error $ErrorMessage
        }
    }
}

Function GatherOutput($AdapterName, $DisplayModules, $DisplayLevels)
{
    $LogConfigs = @()
    $ValidLevels = $IntelLogConfigLevels.Keys
    $LogConfigCount = $DisplayModules.Count
    for ($i = 0; $i -lt $LogConfigCount; $i++)
    {
        $LogConfigs += [PsCustomObject] @{ PSTypeName  = 'IntelLogConfiguration';
                                           Name        = $AdapterName;
                                           Module      = $DisplayModules[$i];
                                           Level       = $DisplayLevels[$i];
                                           ValidLevels = $ValidLevels}
    }

    return $LogConfigs
}

Function GetModuleName($ModuleId)
{
    return $IntelLogConfigModules.GetEnumerator().Where({$_.Value -eq $ModuleId}).Name
}

Function GetLevelName($LevelId)
{
    return $IntelLogConfigLevels.GetEnumerator().Where({$_.Value -eq $LevelId}).Name
}

Function ValidateMethodOutput($MethodOutput, $Verb)
{
    Write-Verbose -Message "MethodOutput: $MethodOutput verb: $Verb"

    do
    {
        $Result = $true
        if ($null -eq $MethodOutput)
        {
            switch ($Verb)
            {
                Get {$script:WarningMessagesGet += $Messages.NoCmdletSupport -f $a; break}
                Set {$script:WarningMessagesSet += $Messages.NoCmdletSupport -f $a; break}
                Start {$script:WarningMessagesStart += $Messages.NoCmdletSupport -f $AdapterName; break}
                Stop {$script:WarningMessagesStop += $Messages.NoCmdletSupport -f $a; break}
                Disable {$script:WarningMessagesDisable += $Messages.NoCmdletSupport -f $a; break}
                Reset {$script:WarningMessagesReset += $Messages.NoCmdletSupport -f $a; break}
            }

            $Result = $false
            break
        }

        if ($MethodOutput.OutStatus -ne 0)
        {
            switch ($Verb)
            {
                Get {$script:ErrorMessagesGet += $Messages.OperationFailed -f $a, $Verb; break}
                Set {$script:ErrorMessagesSet += $Messages.OperationFailed -f $a, $Verb; break}
                Start {$script:ErrorMessagesStart += $Messages.OperationFailed -f $AdapterName, $Verb; break}
                Stop {$script:ErrorMessagesStop += $Messages.OperationFailed -f $a, $Verb; break}
                Disable {$script:ErrorMessagesDisable += $Messages.OperationFailed -f $a, $Verb; break}
                Reset {$script:ErrorMessagesReset += $Messages.OperationFailed -f $a, $Verb; break}
            }

            $Result = $false
        }
    } while ($false)

    Return $Result
}

Function ValidateGetModuleParam([ref]$DisplayModules, [ref]$DisplayLevels, $ModuleNames, $MethodOutput, $AdapterName)
{
    if ($ModuleNames)
    {
        foreach ($ModuleName in $ModuleNames)
        {
            $ModuleIds = $null
            $ModuleIDNames = $IntelLogConfigModules.Keys -Like $ModuleName

            if ($ModuleIDNames.Count -gt 0)
            {
                $ModuleIds = $IntelLogConfigModules[$ModuleIDNames]

                foreach ($ModuleId in $ModuleIds)
                {
                    for ($i = 0; $i -lt $MethodOutput.ModuleID.count; $i++)
                    {
                        if ($ModuleId -eq $MethodOutput.ModuleID[$i])
                        {
                            $DisplayModules.Value += GetModuleName($ModuleId)
                            $DisplayLevels.Value += GetLevelName($MethodOutput.LogLevel[$i])
                            break
                        }
                    }
                }
            }
            else
            {
                #if user passes in an array of modules and one (or more) is spelled incorrectly,
                #show error msg for that 1 and continue on getting the modules for the rest
                $script:ErrorMessagesGet += $Messages.InvalidModule -f $AdapterName, $ModuleName
            }
        }
    }
    else
    {
        GetAllModulesLevel $MethodOutput $DisplayModules $DisplayLevels
    }
}

Function GetAllModulesLevel($MethodOutput, [ref]$DisplayModules, [ref]$DisplayLevels)
{
    for ($i = 0; $i -lt $MethodOutput.ModuleID.count; $i++)
    {
        $DisplayModules.Value += GetModuleName($MethodOutput.ModuleID[$i])
        $DisplayLevels.Value += GetLevelName($MethodOutput.LogLevel[$i])
        # driver returns an array of 64 but only 31 modules populated
        if ($i -eq 31)
        {
            break
        }
    }
}

Function ValidateLevelParam([ref]$InputLevelId, [ref]$DisplayLevels, $LevelName, $Count, $AdapterName)
{
    do
    {
        $NewLevelId = $null
        $Result = $true
        $NewLevelId = @(($IntelLogConfigLevels.GetEnumerator().Where({$_.Name -like $LevelName})).Value)
        if ($NewLevelId.count -gt 1)
        {
            $script:ErrorMessagesSet += $Messages.InvalidLevelWildcard
            $Result = $false
            break
        }

        if (0 -eq $NewLevelId[0].Count)
        {
            $script:ErrorMessagesSet += $Messages.InvalidLevel -f $AdapterName, $LevelName, $ModuleNames[0]
            $Result = $false
            break
        }

        for ($i = 0; $i -lt $Count; $i++)
        {
            $InputLevelId.Value[$i] = $NewLevelId[0]
            $DisplayLevels.Value += GetLevelName($NewLevelId[0])
        }

    } while ($false)

    Return $Result
}

Function ValidateSetModuleParam([ref]$InputModulesId, [ref]$DisplayModules, $ModuleNames, $AdapterName)
{
    $count = 0
    foreach ($ModuleName in $ModuleNames)
    {
        $ModuleIds = $null
        $ModuleIDNames = $IntelLogConfigModules.Keys -Like $ModuleName

        if ($ModuleIDNames.Count -gt 0)
        {
            $ModuleIds = $IntelLogConfigModules[$ModuleIDNames]

            foreach ($ModuleId in $ModuleIds)
            {
                $InputModulesId.Value[$count] = $ModuleId
                $DisplayModules.Value += GetModuleName($ModuleId)
                $count++
            }
        }
        else
        {
            #if user passes in an array of modules and one (or more) is spelled incorrectly,
            #show error msg for that 1 and continue on getting the modules for the rest
            $script:ErrorMessagesSet += $Messages.InvalidModule -f $AdapterName, $ModuleName
        }
    }
}

Function LogmanProcessRunning($AdapterName, $LogName, $GUID)
{
    $Result = $false

    $LogmanQueryString = logman query $LogName -ets
    if ($LogmanQueryString -match ($GUID))
    {
        $script:ErrorMessagesStart += $Messages.LogmanRunning -f $AdapterName
        $Result = $true
    }

    return $Result
}

Function LogmanStart($LogmanPath, $LogName, $GUID)
{
    if ($Append)
    {
        $null = logman start $LogName -ets -o $LogmanPath -p $GUID 0xFFFF 4 -a
    }
    else
    {
        $null = logman start $LogName -ets -o $LogmanPath -p $GUID 0xFFFF 4
    }
}

Function LogmanStop($LogName)
{
    $null = logman stop $LogName -ets
}


Function ValidateLogmanStart()
{
    $Result = $true
    # -2147024629, -2147024773, -2147024893, -2147024891 error codes mean directory name is invalid
    # or filename, directory name, or volume label syntax is incorrect or permission is denied.
    if ($LASTEXITCODE -eq 0)
    {
        $Result = $true
    }
    elseif ($LASTEXITCODE -eq -2147024629 -or $LASTEXITCODE -eq -2147024773 -or $LASTEXITCODE -eq -2147024893 -or $LASTEXITCODE -eq -2147024891)
    {
        $script:ErrorMessagesStart += $Messages.PathIncorrect
        $Result = $false
    }
    else
    {
        $script:ErrorMessagesStart += $Messages.StartCallFailed
        $Result = $false
    }
    return $Result
}

Function ValidateLogmanStop($AdapterName)
{
    $Result = $true

    if ($LASTEXITCODE -eq 0)
    {
        $Result = $true
    }
    elseif ($LASTEXITCODE -eq -2144337918)
    {
        $script:ErrorMessagesStop += $Messages.NoLogsStarted -f $AdapterName
        $Result = $false
    }
    else
    {
        $script:ErrorMessagesStop += $Messages.StopCallFailed
        $Result = $false
    }
    Return $Result
}

Function ValidateConfigurationParam($ConfigurationNames, $ModuleNames, [ref]$DisplayConfigNames, $a)
{
    $Result = $true

    do
    {
        if ($ConfigurationNames)
        {
            if ($ModuleNames)
            {
                $script:ErrorMessagesSet += $Messages.InvalidParamsConfigAndModule
                $script:ErrorMessagesGet += $Messages.InvalidParamsConfigAndModule
                $Result = $false
                break
            }

            foreach ($ConfigName in $ConfigurationNames)
            {
                $TmpDisplayConfigNames = $IntelLogConfigurations.Keys -Like $ConfigName

                if ($TmpDisplayConfigNames.Count -gt 0)
                {
                    $DisplayConfigNames.Value += $TmpDisplayConfigNames
                }
                else
                {
                    #if user passes in an array of configs and one (or more) is spelled incorrectly,
                    #show error msg for that 1 and continue on getting the config for the rest
                    $script:ErrorMessagesSet += $Messages.InvalidConfigurationName -f $a, [string]$ConfigName
                    $script:ErrorMessagesGet += $Messages.InvalidConfigurationName -f $a, [string]$ConfigName
                    $Result = $false
                }
            }
        }
    } while ($false)

    return $Result
}

Function GetConfigLogs($DisplayConfigNames, [ref]$DisplayModules, [ref]$DisplayLevels, $MethodOutput)
{
    $ConfigurationID = $IntelLogConfigurations[$DisplayConfigNames]
    foreach ($ConfigID in $ConfigurationID)
    {
        $params = @{ConfigurationID = [uint32]$ConfigID}
        $GetConfigMethodOutput = InvokeCimMethod "IntlLan_SetGenData" $a "WmiGetFwLogLevelsForConfiguration" $params

        for ($i = 0; $i -lt $GetConfigMethodOutput.count; $i++)
        {
            $DisplayModules.Value += GetModuleName($MethodOutput.ModuleID[$GetConfigMethodOutput.ModuleID[$i]])
            $DisplayLevels.Value += GetLevelName($MethodOutput.LogLevel[$GetConfigMethodOutput.ModuleID[$i]])
        }
    }
}

function GetConfigHashTable($AdapterName, $ConfigPath, [ref]$ErrorMessage)
{
    $ConfigHashTable = @{}
    $FWLogConfigPath = GetAdapterRegistryConfigPath $AdapterName $ConfigPath

    $LogConfigs = (Get-Item $FWLogConfigPath -ErrorAction SilentlyContinue).Property
    if ($null -ne $LogConfigs)
    {
        foreach ($Config in $LogConfigs)
        {
            $ConfigHashTable.add($Config, (Get-ItemProperty -Path $FWLogConfigPath -Name $Config -ErrorAction SilentlyContinue).$Config)
        }
    }
    else
    {
        $TmpErrorMsg = $Messages.NoCmdletSupport -f $AdapterName
        if (-not ($ErrorMessage.Value -contains $TmpErrorMsg))
        {
            $ErrorMessage.Value += $TmpErrorMsg
        }
    }

    return $ConfigHashTable
}

function GetAdapterRegistryConfigPath($AdapterName, $Path)
{
    $ConfigPath = $null
    $AdapterObject = $script:MSNetAdapters.Where({$_.ifDesc -eq $AdapterName})
    $972Key = Get-ItemProperty -Path "HKLM:\SYSTEM\CurrentControlSet\Control\Class\{4d36e972-e325-11ce-bfc1-08002be10318}\*" -Name "NetCfgInstanceId" -ErrorAction SilentlyContinue
    $Adapter972Key = $972Key.Where({$_.NetCfgInstanceId -eq $AdapterObject.InterfaceGuid})
    if ($Adapter972Key)
    {
        $ConfigPath = $Adapter972Key.PSPath + $Path
    }

    return $ConfigPath
}

function GetItemPropertyValue($AdapterName, $Path, $PropertyName, [ref]$ErrorMessage)
{
    $PropertyValue = $null
    $FWLogConfigPath = GetAdapterRegistryConfigPath $AdapterName $Path

    $PropertyItem = Get-ItemProperty -Path $FWLogConfigPath -Name $PropertyName -ErrorAction SilentlyContinue

    if ($null -ne $PropertyItem)
    {
        $PropertyValue = $PropertyItem.$PropertyName
    }
    else
    {
        $TmpErrorMsg = $Messages.NoCmdletSupport -f $AdapterName
        if (-not ($ErrorMessage.Value -contains $TmpErrorMsg))
        {
            $ErrorMessage.Value += $TmpErrorMsg
        }
    }

    return $PropertyValue
}

# SIG # Begin signature block
# MIIorQYJKoZIhvcNAQcCoIIonjCCKJoCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCAAFy8YSEk3uFEi
# X20q+a3OHyx198i0zv/M6k/pyi+5FKCCEgUwggWeMIIEhqADAgECAhEAzS1l4rws
# CIvYBjRVawV4ujANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQGEwJHQjEbMBkGA1UE
# CBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYDVQQHEwdTYWxmb3JkMRgwFgYDVQQK
# Ew9TZWN0aWdvIExpbWl0ZWQxJDAiBgNVBAMTG1NlY3RpZ28gUlNBIENvZGUgU2ln
# bmluZyBDQTAeFw0yMTA0MDIwMDAwMDBaFw0yMzA0MDIyMzU5NTlaMIGEMQswCQYD
# VQQGEwJVUzETMBEGA1UECAwKQ2FsaWZvcm5pYTEUMBIGA1UEBwwLU2FudGEgQ2xh
# cmExGjAYBgNVBAoMEUludGVsIENvcnBvcmF0aW9uMRIwEAYDVQQLDAlTb2xhcktN
# Q1MxGjAYBgNVBAMMEUludGVsIENvcnBvcmF0aW9uMIIBojANBgkqhkiG9w0BAQEF
# AAOCAY8AMIIBigKCAYEA7CCN9iKpDHOrRceKhlXFP/tf6Lllw2H2fR9KVI4/fQIx
# MU1hXwnlHmAzMCY7IgcCFY4p3F5/MJGKaqYngwOo28Zo6Q1N6ukysA7PSavmF2RY
# WD6VFeya/2H0PoNeRFjHaRzSeynFFeJAFew9r7UReUwM/507sxZYPQuWWIdAEK7H
# Dqp2VlHmgZOXVGHhNO6GFOKpC/C01g6X3x6OquddRNMt5UrZzZzDo5MpJz9SBB2V
# jiqwZ80dvNR2W2xi90cIHh4BkXvB54UNkp4VTVu16T0k3cweo+C39U7GrCAr5Axz
# DETjBvhNtP1sf9SoRV7xY6g5wssfI7yYT9J0gsifn/Vy8MWH355TPoA+PVhbAu0m
# 9FMz4EWu55nnUurNML2jaUxsos21/7ELat12kWC0tq9fhkODjKO8X9PuiBHflZLk
# d3F4QcSMvuGocWGqE77VV3vn8jlvigm2TOV0CfGTQajGMX0jeTRZ19fzBNkt2X9d
# SSGolI/Kj1gSvCggpkUBAgMBAAGjggGQMIIBjDAfBgNVHSMEGDAWgBQO4TqoUzox
# 1Yq+wbutZxoDha00DjAdBgNVHQ4EFgQUshkNuM2SdwJnW4vFy8c4FtUTrbQwDgYD
# VR0PAQH/BAQDAgeAMAwGA1UdEwEB/wQCMAAwEwYDVR0lBAwwCgYIKwYBBQUHAwMw
# EQYJYIZIAYb4QgEBBAQDAgQQMEoGA1UdIARDMEEwNQYMKwYBBAGyMQECAQMCMCUw
# IwYIKwYBBQUHAgEWF2h0dHBzOi8vc2VjdGlnby5jb20vQ1BTMAgGBmeBDAEEATBD
# BgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29S
# U0FDb2RlU2lnbmluZ0NBLmNybDBzBggrBgEFBQcBAQRnMGUwPgYIKwYBBQUHMAKG
# Mmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1JTQUNvZGVTaWduaW5nQ0Eu
# Y3J0MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG
# 9w0BAQsFAAOCAQEAVadLNRW4f/pKMqrbn0BdOoQ8/1EJ87gvVfosei2bLwTEvpmv
# mn2n561H6AFedtIJ6L4FmXII4M4r20i+5LREbI6PpKDmOAf4xW7POxfCRvkTQAZO
# 3zoVxjMQBXo7cZVF1xHCdviXzD1usuIiCF8DLm6z4O/kyeFFNcn816yPQct91Pnk
# SBBVvL+Kwu8xvR+ZIQy632WUA4HnNpRdFnVSzUifEg2GrtsKZR8k+rm2o8K8yjJq
# 3SznwgJQCMVMh3CtRtUwE/c7o/6rvm53fTYJDd3aoPHVgH6S2WqS3+3mQG7A6hTD
# nrP/mYnS4PF7XzxxjZhUlhy4G/MarJPvT9IrNDCCBfUwggPdoAMCAQICEB2iSDBv
# myYY0ILgln0z02owDQYJKoZIhvcNAQEMBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0eTEeMBwGA1UEChMV
# VGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1c3QgUlNBIENl
# cnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTE4MTEwMjAwMDAwMFoXDTMwMTIzMTIz
# NTk1OVowfDELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFuY2hlc3Rl
# cjEQMA4GA1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSQw
# IgYDVQQDExtTZWN0aWdvIFJTQSBDb2RlIFNpZ25pbmcgQ0EwggEiMA0GCSqGSIb3
# DQEBAQUAA4IBDwAwggEKAoIBAQCGIo0yhXoYn0nwli9jCB4t3HyfFM/jJrYlZilA
# hlRGdDFixRDtsocnppnLlTDAVvWkdcapDlBipVGREGrgS2Ku/fD4GKyn/+4uMyD6
# DBmJqGx7rQDDYaHcaWVtH24nlteXUYam9CflfGqLlR5bYNV+1xaSnAAvaPeX7Wpy
# vjg7Y96Pv25MQV0SIAhZ6DnNj9LWzwa0VwW2TqE+V2sfmLzEYtYbC43HZhtKn52B
# xHJAteJf7wtF/6POF6YtVbC3sLxUap28jVZTxvC6eVBJLPcDuf4vZTXyIuosB69G
# 2flGHNyMfHEo8/6nxhTdVZFuihEN3wYklX0Pp6F8OtqGNWHTAgMBAAGjggFkMIIB
# YDAfBgNVHSMEGDAWgBRTeb9aqitKz1SA4dibwJ3ysgNmyzAdBgNVHQ4EFgQUDuE6
# qFM6MdWKvsG7rWcaA4WtNA4wDgYDVR0PAQH/BAQDAgGGMBIGA1UdEwEB/wQIMAYB
# Af8CAQAwHQYDVR0lBBYwFAYIKwYBBQUHAwMGCCsGAQUFBwMIMBEGA1UdIAQKMAgw
# BgYEVR0gADBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8vY3JsLnVzZXJ0cnVzdC5j
# b20vVVNFUlRydXN0UlNBQ2VydGlmaWNhdGlvbkF1dGhvcml0eS5jcmwwdgYIKwYB
# BQUHAQEEajBoMD8GCCsGAQUFBzAChjNodHRwOi8vY3J0LnVzZXJ0cnVzdC5jb20v
# VVNFUlRydXN0UlNBQWRkVHJ1c3RDQS5jcnQwJQYIKwYBBQUHMAGGGWh0dHA6Ly9v
# Y3NwLnVzZXJ0cnVzdC5jb20wDQYJKoZIhvcNAQEMBQADggIBAE1jUO1HNEphpNve
# aiqMm/EAAB4dYns61zLC9rPgY7P7YQCImhttEAcET7646ol4IusPRuzzRl5ARokS
# 9At3WpwqQTr81vTr5/cVlTPDoYMot94v5JT3hTODLUpASL+awk9KsY8k9LOBN9O3
# ZLCmI2pZaFJCX/8E6+F0ZXkI9amT3mtxQJmWunjxucjiwwgWsatjWsgVgG10Xkp1
# fqW4w2y1z99KeYdcx0BNYzX2MNPPtQoOCwR/oEuuu6Ol0IQAkz5TXTSlADVpbL6f
# ICUQDRn7UJBhvjmPeo5N9p8OHv4HURJmgyYZSJXOSsnBf/M6BZv5b9+If8AjntIe
# Q3pFMcGcTanwWbJZGehqjSkEAnd8S0vNcL46slVaeD68u28DECV3FTSK+TbMQ5Lk
# uk/xYpMoJVcp+1EZx6ElQGqEV8aynbG8HArafGd+fS7pKEwYfsR7MUFxmksp7As9
# V1DSyt39ngVR5UR43QHesXWYDVQk/fBO4+L4g71yuss9Ou7wXheSaG3IYfmm8SoK
# C6W59J7umDIFhZ7r+YMp08Ysfb06dy6LN0KgaoLtO0qqlBCk4Q34F8W2WnkzGJLj
# tXX4oemOCiUe5B7xn1qHI/+fpFGe+zmAEc3btcSnqIBv5VPU4OOiwtJbGvoyJi1q
# V3AcPKRYLqPzW0sH3DJZ84enGm1YMIIGZjCCBE6gAwIBAgITMwAAAES3P/zvWs+i
# egAAAAAARDANBgkqhkiG9w0BAQUFADB/MQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMSkwJwYDVQQDEyBNaWNyb3NvZnQgQ29kZSBWZXJpZmljYXRp
# b24gUm9vdDAeFw0xNTA3MjIyMTAzNDlaFw0yNTA3MjIyMTAzNDlaMIGIMQswCQYD
# VQQGEwJVUzETMBEGA1UECBMKTmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENp
# dHkxHjAcBgNVBAoTFVRoZSBVU0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNF
# UlRydXN0IFJTQSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eTCCAiIwDQYJKoZIhvcN
# AQEBBQADggIPADCCAgoCggIBAIASZRc2DsPbCLPQrFcNdu3NJ9NMrVCDYeKqIE0J
# LWQJ3M6Jn8w9qez2z8Hc8dOx1ns3KBErR9o5xrw6GbRfpr19naNjQrZ28qk7K5H4
# 4m/Q7BYgkAk+4uh0yRi0kdRiZNt/owbxiBhqkCI8vP4T8IcUe/bkH47U5FHGEWdG
# CFHLhhRUP7wz/n5snP8WnRi9UY41pqdmyHJn2yFmsdSbeAPAUDrozPDcvJ5M/q8F
# ljUfV1q3/875PbcstvZU3cjnEjpNrkyKt1yatLcgPcp/IjSufjtoZgFE5wFORlOb
# M2D3lL5TN5BzQ/Myw1Pv26r+dE5px2uMYJPexMcM3+EyrsyTO1F4lWeL7j1W/gzQ
# aQ8bD/MlJmszbfduR/pzQ+V+DqVmsSl8MoRjVYnEDcGTVDAZE6zTfTen6106bDVc
# 20HXEtqpSQvf2ICKCZNijrVmzyWIzYS4sT+kOQ/ZAp7rEkyVfPNrBaleFoPMuGfi
# 6BOdzFuC00yz7Vv/3uVzrCM7LQC/NVV0CUnYSVgaf5I25lGSDvMmfRxNF7zJ7EMm
# 0L9BX0CpRET0medXh55QH1dUqD79dGMvsVBlCeZYQi5DGky08CVHWfoEHpPUJkZK
# UIGy3r54t/xnFeHJV4QeD2PW6WK61l9VLupcxigIBCU5uA4rqfJMlxwHPw1S9e3v
# L4IPAgMBAAGjgdAwgc0wEwYDVR0lBAwwCgYIKwYBBQUHAwMwEgYDVR0TAQH/BAgw
# BgEB/wIBAjAdBgNVHQ4EFgQUU3m/WqorSs9UgOHYm8Cd8rIDZsswCwYDVR0PBAQD
# AgGGMB8GA1UdIwQYMBaAFGL7CiFbf0NuEdoJVFBr9dKWcfGeMFUGA1UdHwROMEww
# SqBIoEaGRGh0dHA6Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3Rz
# L01pY3Jvc29mdENvZGVWZXJpZlJvb3QuY3JsMA0GCSqGSIb3DQEBBQUAA4ICAQBr
# IpM8PTlUcWRrDvLkPDARxSBKS4YPkvH/M3k62eSYpw5AoCKAfmGy4KcZzyaVMSpl
# 1GpPMYbqwMYuxWSMPUhZzQsvdD2UJhMQQtSXmCdePHbSeGkdGmTnBXJ14OtmQEOf
# jwxG/5dgpshnrRAIm2Km6b46itMHTZ9ykyW8BhHgLJA4Pmcc/RnXnpDOPcLg52Gs
# wOUE9R6ZVAyRDQFWcTeuJ9SeQyKlySfNTeVxEjkkpUFWh/+8VRQPJcqJ7seX5dIT
# /z1+GqCPP8gs16Nw0MdgwPzYPlHnl8Y+O+3PeL6KyuPE8qen7Z6uCAKPoFLbch7V
# O8NNn476m3DH+OO/bD+Sm+Q3PuxqjCn5waK/iz4aaWb7HGNPJgHJAsQ+0v/DQ6gb
# /Zn61LylueKTLzsBxdH0Oi9ow+Bkt1qVXkbMB4NpuzwFklZzNXNFmE582BKlt0Lp
# omP2QmAYcNE7bzHAh8fmceHzRhbp9bhys+ltH2ImSaNJi91ox4toVvfe/PqHJLgD
# gReP5fFnah2u03T3jKVdswuOQimWzknEd35mfAEXGmwUJMOwF3cF2BpAt4Zr2OR7
# QKx+305vJPkggIKMM+fl+inYndqLcF0ryR2CTAtny4RBnucGfhGDRC2KGe70f5rd
# eRw3GR6fP4wpug1cEIY3bEjNRV3NcLy80U1d2MW4djGCFf4wghX6AgEBMIGRMHwx
# CzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNV
# BAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEkMCIGA1UEAxMb
# U2VjdGlnbyBSU0EgQ29kZSBTaWduaW5nIENBAhEAzS1l4rwsCIvYBjRVawV4ujAN
# BglghkgBZQMEAgEFAKBqMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisG
# AQQBgjcCAQsxDjAMBgorBgEEAYI3AgEWMC8GCSqGSIb3DQEJBDEiBCBD6YOkHH6n
# JyPP7QVnxJZb39P2OmRqlEiNI0LuEHxe4jANBgkqhkiG9w0BAQEFAASCAYBtE0kx
# oO050Swj/TLRx2dzO1FgS0NrcBeu4Zm+mR2OwkupgkoYoaN5JmDBWKPjibeo4hHD
# 5CJ+M0sXI8rgk3DObBKBuFpNORMQyGipj/+kv247wTIgN5ecg9yh21xqOoSygeCV
# SZ4MTY2ZLYBbt3Q4oG4FcyRag+3mPqHnUqGY3oBHFPsGtHhukWzm5/Rr9uaM7dPw
# VheiP/deu+x9gOBOi9EYn8L9I2nb9tT69Wst9iliLBNDiR3P3Fp9lFWXM7eycaPe
# 4QsiB14WGRu4Zvm/hTVxXvC3RsFlh8HxfSwBR/WXtEHgGW6qPAxssVMMZ1YTj64Y
# MQxN3gY2/MMXgD9WkUuAvUTiMEhhorG2BwkLD76N6KLd2t+ZWgZD8R+otTFQhPgI
# m91NBz3pidQ2X4/C/VAJo8NlK7GW8QxTtGvBgpKZGFeP74n5mpDa4JRaN1aYhTn1
# XlhwR53UwurlKP4HAsBuhVwByKOTNHU2VnmCGxGdOzrZjuEmLetXkaYFo9ihghNR
# MIITTQYKKwYBBAGCNwMDATGCEz0wghM5BgkqhkiG9w0BBwKgghMqMIITJgIBAzEP
# MA0GCWCGSAFlAwQCAgUAMIHwBgsqhkiG9w0BCRABBKCB4ASB3TCB2gIBAQYKKwYB
# BAGyMQIBATAxMA0GCWCGSAFlAwQCAQUABCD73SRjjJuyfbMKKubNLmokPt9s02AE
# A3MoUtVEQ+mLdwIVAOPhijRIcHN1LrL20QDlfB7lydiLGA8yMDIzMDExMDIzMDAw
# NlqgbqRsMGoxCzAJBgNVBAYTAkdCMRMwEQYDVQQIEwpNYW5jaGVzdGVyMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLDAqBgNVBAMMI1NlY3RpZ28gUlNBIFRpbWUg
# U3RhbXBpbmcgU2lnbmVyICMzoIIN6jCCBvYwggTeoAMCAQICEQCQOX+a0ko6E/K9
# kV8IOKlDMA0GCSqGSIb3DQEBDAUAMH0xCzAJBgNVBAYTAkdCMRswGQYDVQQIExJH
# cmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcTB1NhbGZvcmQxGDAWBgNVBAoTD1Nl
# Y3RpZ28gTGltaXRlZDElMCMGA1UEAxMcU2VjdGlnbyBSU0EgVGltZSBTdGFtcGlu
# ZyBDQTAeFw0yMjA1MTEwMDAwMDBaFw0zMzA4MTAyMzU5NTlaMGoxCzAJBgNVBAYT
# AkdCMRMwEQYDVQQIEwpNYW5jaGVzdGVyMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0
# ZWQxLDAqBgNVBAMMI1NlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcgU2lnbmVyICMz
# MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAkLJxP3nh1LmKF8zDl8KQ
# lHLtWjpvAUN/c1oonyR8oDVABvqUrwqhg7YT5EsVBl5qiiA0cXu7Ja0/WwqkHy9s
# fS5hUdCMWTc+pl3xHl2AttgfYOPNEmqIH8b+GMuTQ1Z6x84D1gBkKFYisUsZ0vCW
# yUQfOV2csJbtWkmNfnLkQ2t/yaA/bEqt1QBPvQq4g8W9mCwHdgFwRd7D8EJp6v8m
# zANEHxYo4Wp0tpxF+rY6zpTRH72MZar9/MM86A2cOGbV/H0em1mMkVpCV1VQFg1L
# dHLuoCox/CYCNPlkG1n94zrU6LhBKXQBPw3gE3crETz7Pc3Q5+GXW1X3KgNt1c1i
# 2s6cHvzqcH3mfUtozlopYdOgXCWzpSdoo1j99S1ryl9kx2soDNqseEHeku8Pxeyr
# 3y1vGlRRbDOzjVlg59/oFyKjeUFiz/x785LaruA8Tw9azG7fH7wir7c4EJo0pwv/
# /h1epPPuFjgrP6x2lEGdZB36gP0A4f74OtTDXrtpTXKZ5fEyLVH6Ya1N6iaObfyp
# SJg+8kYNabG3bvQF20EFxhjAUOT4rf6sY2FHkbxGtUZTbMX04YYnk4Q5bHXgHQx6
# WYsuy/RkLEJH9FRYhTflx2mn0iWLlr/GreC9sTf3H99Ce6rrHOnrPVrd+NKQ1Uma
# Oh2DGld/HAHCzhx9zPuWFcUCAwEAAaOCAYIwggF+MB8GA1UdIwQYMBaAFBqh+GEZ
# IA/DQXdFKI7RNV8GEgRVMB0GA1UdDgQWBBQlLmg8a5orJBSpH6LfJjrPFKbx4DAO
# BgNVHQ8BAf8EBAMCBsAwDAYDVR0TAQH/BAIwADAWBgNVHSUBAf8EDDAKBggrBgEF
# BQcDCDBKBgNVHSAEQzBBMDUGDCsGAQQBsjEBAgEDCDAlMCMGCCsGAQUFBwIBFhdo
# dHRwczovL3NlY3RpZ28uY29tL0NQUzAIBgZngQwBBAIwRAYDVR0fBD0wOzA5oDeg
# NYYzaHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdvUlNBVGltZVN0YW1waW5n
# Q0EuY3JsMHQGCCsGAQUFBwEBBGgwZjA/BggrBgEFBQcwAoYzaHR0cDovL2NydC5z
# ZWN0aWdvLmNvbS9TZWN0aWdvUlNBVGltZVN0YW1waW5nQ0EuY3J0MCMGCCsGAQUF
# BzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG9w0BAQwFAAOCAgEA
# c9rtaHLLwrlAoTG7tAOjLRR7JOe0WxV9qOn9rdGSDXw9NqBp2fOaMNqsadZ0VyQ/
# fg882fXDeSVsJuiNaJPO8XeJOX+oBAXaNMMU6p8IVKv/xH6WbCvTlOu0bOBFTSyy
# 9zs7WrXB+9eJdW2YcnL29wco89Oy0OsZvhUseO/NRaAA5PgEdrtXxZC+d1SQdJ4L
# T03EqhOPl68BNSvLmxF46fL5iQQ8TuOCEmLrtEQMdUHCDzS4iJ3IIvETatsYL254
# rcQFtOiECJMH+X2D/miYNOR35bHOjJRs2wNtKAVHfpsu8GT726QDMRB8Gvs8GYDR
# C3C5VV9HvjlkzrfaI1Qy40ayMtjSKYbJFV2Ala8C+7TRLp04fDXgDxztG0dInCJq
# VYLZ8roIZQPl8SnzSIoJAUymefKithqZlOuXKOG+fRuhfO1WgKb0IjOQ5IRT/Cr6
# wKeXqOq1jXrO5OBLoTOrC3ag1WkWt45mv1/6H8Sof6ehSBSRDYL8vU2Z7cnmbDb+
# d0OZuGktfGEv7aOwSf5bvmkkkf+T/FdpkkvZBT9thnLTotDAZNI6QsEaA/vQ7Zoh
# uD+vprJRVNVMxcofEo1XxjntXP/snyZ2rWRmZ+iqMODSrbd9sWpBJ24DiqN04IoJ
# gm6/4/a3vJ4LKRhogaGcP24WWUsUCQma5q6/YBXdhvUwggbsMIIE1KADAgECAhAw
# D2+s3WaYdHypRjaneC25MA0GCSqGSIb3DQEBDAUAMIGIMQswCQYDVQQGEwJVUzET
# MBEGA1UECBMKTmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENpdHkxHjAcBgNV
# BAoTFVRoZSBVU0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNFUlRydXN0IFJT
# QSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eTAeFw0xOTA1MDIwMDAwMDBaFw0zODAx
# MTgyMzU5NTlaMH0xCzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNo
# ZXN0ZXIxEDAOBgNVBAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRl
# ZDElMCMGA1UEAxMcU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBDQTCCAiIwDQYJ
# KoZIhvcNAQEBBQADggIPADCCAgoCggIBAMgbAa/ZLH6ImX0BmD8gkL2cgCFUk7nP
# oD5T77NawHbWGgSlzkeDtevEzEk0y/NFZbn5p2QWJgn71TJSeS7JY8ITm7aGPwEF
# kmZvIavVcRB5h/RGKs3EWsnb111JTXJWD9zJ41OYOioe/M5YSdO/8zm7uaQjQqzQ
# FcN/nqJc1zjxFrJw06PE37PFcqwuCnf8DZRSt/wflXMkPQEovA8NT7ORAY5unSd1
# VdEXOzQhe5cBlK9/gM/REQpXhMl/VuC9RpyCvpSdv7QgsGB+uE31DT/b0OqFjIpW
# cdEtlEzIjDzTFKKcvSb/01Mgx2Bpm1gKVPQF5/0xrPnIhRfHuCkZpCkvRuPd25Ff
# nz82Pg4wZytGtzWvlr7aTGDMqLufDRTUGMQwmHSCIc9iVrUhcxIe/arKCFiHd6QV
# 6xlV/9A5VC0m7kUaOm/N14Tw1/AoxU9kgwLU++Le8bwCKPRt2ieKBtKWh97oaw7w
# W33pdmmTIBxKlyx3GSuTlZicl57rjsF4VsZEJd8GEpoGLZ8DXv2DolNnyrH6jaFk
# yYiSWcuoRsDJ8qb/fVfbEnb6ikEk1Bv8cqUUotStQxykSYtBORQDHin6G6UirqXD
# TYLQjdprt9v3GEBXc/Bxo/tKfUU2wfeNgvq5yQ1TgH36tjlYMu9vGFCJ10+dM70a
# tZ2h3pVBeqeDAgMBAAGjggFaMIIBVjAfBgNVHSMEGDAWgBRTeb9aqitKz1SA4dib
# wJ3ysgNmyzAdBgNVHQ4EFgQUGqH4YRkgD8NBd0UojtE1XwYSBFUwDgYDVR0PAQH/
# BAQDAgGGMBIGA1UdEwEB/wQIMAYBAf8CAQAwEwYDVR0lBAwwCgYIKwYBBQUHAwgw
# EQYDVR0gBAowCDAGBgRVHSAAMFAGA1UdHwRJMEcwRaBDoEGGP2h0dHA6Ly9jcmwu
# dXNlcnRydXN0LmNvbS9VU0VSVHJ1c3RSU0FDZXJ0aWZpY2F0aW9uQXV0aG9yaXR5
# LmNybDB2BggrBgEFBQcBAQRqMGgwPwYIKwYBBQUHMAKGM2h0dHA6Ly9jcnQudXNl
# cnRydXN0LmNvbS9VU0VSVHJ1c3RSU0FBZGRUcnVzdENBLmNydDAlBggrBgEFBQcw
# AYYZaHR0cDovL29jc3AudXNlcnRydXN0LmNvbTANBgkqhkiG9w0BAQwFAAOCAgEA
# bVSBpTNdFuG1U4GRdd8DejILLSWEEbKw2yp9KgX1vDsn9FqguUlZkClsYcu1UNvi
# ffmfAO9Aw63T4uRW+VhBz/FC5RB9/7B0H4/GXAn5M17qoBwmWFzztBEP1dXD4rzV
# WHi/SHbhRGdtj7BDEA+N5Pk4Yr8TAcWFo0zFzLJTMJWk1vSWVgi4zVx/AZa+clJq
# O0I3fBZ4OZOTlJux3LJtQW1nzclvkD1/RXLBGyPWwlWEZuSzxWYG9vPWS16toytC
# iiGS/qhvWiVwYoFzY16gu9jc10rTPa+DBjgSHSSHLeT8AtY+dwS8BDa153fLnC6N
# Ixi5o8JHHfBd1qFzVwVomqfJN2Udvuq82EKDQwWli6YJ/9GhlKZOqj0J9QVst9Jk
# WtgqIsJLnfE5XkzeSD2bNJaaCV+O/fexUpHOP4n2HKG1qXUfcb9bQ11lPVCBbqvw
# 0NP8srMftpmWJvQ8eYtcZMzN7iea5aDADHKHwW5NWtMe6vBE5jJvHOsXTpTDeGUg
# Ow9Bqh/poUGd/rG4oGUqNODeqPk85sEwu8CgYyz8XBYAqNDEf+oRnR4GxqZtMl20
# OAkrSQeq/eww2vGnL8+3/frQo4TZJ577AWZ3uVYQ4SBuxq6x+ba6yDVdM3aO8Xwg
# DCp3rrWiAoa6Ke60WgCxjKvj+QrJVF3UuWp0nr1IrpgxggQtMIIEKQIBATCBkjB9
# MQswCQYDVQQGEwJHQjEbMBkGA1UECBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYD
# VQQHEwdTYWxmb3JkMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxJTAjBgNVBAMT
# HFNlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcgQ0ECEQCQOX+a0ko6E/K9kV8IOKlD
# MA0GCWCGSAFlAwQCAgUAoIIBazAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQAQQw
# HAYJKoZIhvcNAQkFMQ8XDTIzMDExMDIzMDAwNlowPwYJKoZIhvcNAQkEMTIEMKPd
# MGFJCCgvMbxYVINb+OQMb/VoYc2cMCHSiN3XtSWlgwoZoE1aHixwuUFK0LXRRDCB
# 7QYLKoZIhvcNAQkQAgwxgd0wgdowgdcwFgQUqzQBOqxAlzGfCBrwsxjhg/gPeIEw
# gbwEFALWW5Xig3DBVwCV+oj5I92Tf62PMIGjMIGOpIGLMIGIMQswCQYDVQQGEwJV
# UzETMBEGA1UECBMKTmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENpdHkxHjAc
# BgNVBAoTFVRoZSBVU0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNFUlRydXN0
# IFJTQSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eQIQMA9vrN1mmHR8qUY2p3gtuTAN
# BgkqhkiG9w0BAQEFAASCAgBXSRoDqP48+5hncB4AlNKemN/vnQeT+/KjW3PQdB2b
# aj7n3gMc4fn3W2ND0cn83DQbAz+nVxkZOqs+p4de1MHKYlV7sXFEOqXh82HBa8Kz
# e6yOpSaJ+OdBOLG8s5HDWXH3krt3jA+WhzGYkLrt/C0b5y8uM6t2NP38WSjcSYth
# oLFCUxYmFanaCY6suBi2RhuJedQmjGQRYqDdZpZRVkCqxfwVEnF4RHkhS9359RsL
# 4zcMb0CQmPsglALJVRatjNni5JqQ9jOMeCylZrsUvHTfTkCLskY0UPxtsh19Hzbm
# bwWf4IP454XyleQ4qIwip4QPmfnjd72mxdGxzumf3LH+hRkrNzCfYfkHzCXC1Wp+
# Y9gQIzzwJLtvlr45tpIjM/9DYa/Vuxra3QW1cF3l8yrfbXOr6fFdrSV6am3cMT+v
# NxppRTMAY1NqCSi79R9TPio6g4cjs59iidyVk37dz6POlL/PMH5HzzUS5+IGbldU
# tL8DHnHBZ5dfH7jznT2U1ZyMqX3/ZelifTSCIv6wIm6MTenVyXoe8ieO6bVW/vJa
# JJrH8X900r0JjV0sV9xOB8OqR6KV/jK93VmPX2RhC6v86X2dPO0l4/S1Afg0V8Eg
# 2owoYMQpgo4BgcBMW4Ct1BAX3adJFI5WHUQgOrVj9d5c3kRkBsPPXWU0jZnQo6gJ
# 2Q==
# SIG # End signature block
