# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2022 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSAvoidUsingPositionalParameters", "", Scope="function")]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSUseShouldProcessForStateChangingFunctions", "", Scope="function")] Param()

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Set-IntelEthernetSetting
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory = $false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory = $false, Position = 0, ValueFromPipeline = $true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null,
    [parameter(Mandatory = $false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $DisplayName = '',
    [parameter(Mandatory = $false)]
    [ValidateNotNullOrEmpty()]
    [object]
    $DisplayValue = $null,
    [parameter(Mandatory = $false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $RegistryKeyword = '',
    [parameter(Mandatory = $false)]
    [ValidateNotNullOrEmpty()]
    [object]
    $RegistryValue = $null,
    [parameter(Mandatory = $false)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Setting = $null,
    [parameter(Mandatory = $false)]
    [switch]
    $Reset
    )
    Begin
    {
        $AdapterName = $Name
        $script:ErrorMessagesSet = @()
        $script:WarningMessagesSet = @()
        $FinalObject = @()
        GetAllSupportedEthernetDevices
        $script:MSNetAdvProperty = Get-NetAdapterAdvancedProperty -ErrorAction SilentlyContinue
    }
    Process
    {
        $Adapters = $Adapter

        if (IsSettingObject $Adapters)
        {
            $Setting = $Adapters
            $Adapters = $null
        }

        do
        {
            if (!(ValidateParams $DisplayName $DisplayValue $RegistryKeyword $RegistryValue $Setting $AdapterName $Adapters))
            {
                break
            }

            if ($null -ne $Setting)
            {
                $PreProcessedAdapterNames = ValidateSetAdapterNameParams $Setting.Name $null ([ref]$script:ErrorMessagesSet)
                $AdapterNamesArray = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:ErrorMessagesSet) | Get-Unique)
            }
            else
            {
                $PreProcessedAdapterNames = ValidateSetAdapterNameParams $AdapterName $Adapters ([ref]$script:ErrorMessagesSet)
                $AdapterNamesArray = @(GetSupportedAdapters $PreProcessedAdapterNames ([ref]$script:ErrorMessagesSet))
            }

            try
            {
                $PreErrorActionPreference = $global:ErrorActionPreference
                $global:ErrorActionPreference = 'SilentlyContinue'
                foreach ($a in $AdapterNamesArray)
                {
                    $TmpStatusMsg = CheckDeviceError $a
                    if (-not [string]::IsNullOrEmpty($TmpStatusMsg))
                    {
                        $script:ErrorMessagesSet += $TmpStatusMsg
                    }

                    $AdapterSettings = @()
                    if ($null -ne $Setting)
                    {
                        $AdapterSettings = GetSetSettings $null $Setting.RegistryKeyword $a
                    }
                    else
                    {
                        $AdapterSettings = GetSetSettings $DisplayName $RegistryKeyword $a
                    }

                    if ($AdapterSettings.RegistryKeyword -Contains 'NetworkAddress' -and -not $Reset)
                    {
                        if ($false -eq (ValidateLAASetting $DisplayValue $RegistryValue))
                        {
                            break
                        }
                    }

                    foreach ($adapterSetting in $AdapterSettings)
                    {
                        if ($null -ne $DisplayValue)
                        {
                            Set-NetAdapterAdvancedProperty -InterfaceDescription $a -DisplayName $adapterSetting.DisplayName -DisplayValue $DisplayValue
                            $FinalObject += ValidateSetting $adapterSetting $DisplayValue $null $a
                        }
                        elseif ($null -ne $RegistryValue)
                        {
                            Set-NetAdapterAdvancedProperty -InterfaceDescription $a -DisplayName $adapterSetting.DisplayName -RegistryValue $RegistryValue
                            $FinalObject += ValidateSetting $adapterSetting $null $RegistryValue $a
                        }
                        elseif ($Reset)
                        {
                            Reset-NetAdapterAdvancedProperty -InterfaceDescription $a -DisplayName $adapterSetting.DisplayName
                            $FinalObject += ValidateSetting $adapterSetting $adapterSetting.DefaultDisplayValue $null $a
                        }
                    }
                }
            }
            finally
            {
                $global:ErrorActionPreference = $PreErrorActionPreference
            }
        } while ($false)
    }
    End
    {
        $FinalObject

        foreach ($WarningMessage in $script:WarningMessagesSet)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesSet)
        {
            Write-Error $ErrorMessage
        }
    }
}

function ValidateParams($DisplayName, $DisplayValue, $RegistryKeyword, $RegistryValue, $Setting, $AdapterName, $Adapters)
{
    $bContinue = $true

    do
    {
        # if DisplayName, RegistryKeyword, & Setting are not used, or both DisplayName & RegistryKeyword are used - stop execution
        if (([string]::IsNullOrEmpty($DisplayName) -and [string]::IsNullOrEmpty($RegistryKeyword) -and $null -eq $Setting) -or
            (-not [string]::IsNullOrEmpty($DisplayName) -and -not [string]::IsNullOrEmpty($RegistryKeyword)))
        {
            $bContinue = $false
            break
        }

        # if both DisplayValue & RegistryValue are not used & Reset isn't used, or both DisplayValue & RegistryValue are used - stop exection
        if (([string]::IsNullOrEmpty($DisplayValue) -and [string]::IsNullOrEmpty($RegistryValue) -and -not $Reset) -or
            (-not [string]::IsNullOrEmpty($DisplayValue) -and -not [string]::IsNullOrEmpty($RegistryValue)))
        {
            $bContinue = $false
            break
        }

        # if either DisplayValue & RegistryValue are used and $Reset is used - stop execution
        if (-not [string]::IsNullOrEmpty($DisplayValue) -or -not [string]::IsNullOrEmpty($RegistryValue) -and $Reset)
        {
            $bContinue = $false
            break
        }

        # if Setting is used and either DisplayName, RegistryKeyword are used - stop execution
        if ($null -ne $Setting -and (-not [string]::IsNullOrEmpty($DisplayName) -or -not [string]::IsNullOrEmpty($RegistryKeyword)))
        {
            $bContinue = $false
            break
        }

        # if Setting is used and either $AdapterName or $Adapters is used - stop execution
        if ($null -ne $Setting -and (-not [string]::IsNullOrEmpty($AdapterName) -or $Adapters))
        {
            $bContinue = $false
            break
        }

    } while ($false)

    if ($false -eq $bContinue)
    {
        $script:ErrorMessagesSet += $Messages.InvalidParams
    }

    return $bContinue
}

function GetSetSettings($DisplayName, $RegistryKeyword, $AdapterName)
{
    $SettingArray = @()

    if (-not [string]::IsNullOrEmpty($DisplayName))
    {
        foreach ($TmpDisplayName in $DisplayName)
        {
            $TmpSetting = $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName -and $_.DisplayName -like $TmpDisplayName})
            if (-not $TmpSetting)
            {
                $script:ErrorMessagesSet += $Messages.InvalidSetting -f $AdapterName, $TmpDisplayName
            }
            else
            {
                $SettingArray += $TmpSetting
            }
        }
    }

    if (-not [string]::IsNullOrEmpty($RegistryKeyword))
    {
        foreach ($TmpRegistryKeyword in $RegistryKeyword)
        {
            $TmpSetting = $script:MSNetAdvProperty.Where({$_.InterfaceDescription -eq $AdapterName -and $_.RegistryKeyword -like $TmpRegistryKeyword})
            if (-not $TmpSetting)
            {
                $script:ErrorMessagesSet += $Messages.InvalidSetting -f $AdapterName, $TmpRegistryKeyword
            }
            else
            {
                $SettingArray += $TmpSetting
            }
        }
    }

    return $SettingArray
}

function ValidateSetting($Setting, $DisplayValue = $null, $RegistryValue = $null, $AdapterName)
{
    $IntelEthernetSetting = Get-IntelEthernetSetting -Name $a -DisplayName $Setting.DisplayName -WarningAction:SilentlyContinue
    # setting type 'int'
    if ($null -ne $IntelEthernetSetting.PSTypeNames -and $IntelEthernetSetting.PSTypeNames[0] -eq "IntelEthernetSettingInt")
    {
        if ($null -ne $DisplayValue)
        {
            $SettingIntValue = $DisplayValue
        }
        else
        {
            $SettingIntValue = $RegistryValue
        }

        $bValidIntValue = $false
        # check if numeric value
        if ($SettingIntValue -match "^[\d]+$")
        {
            if ([int]$SettingIntValue -ge $IntelEthernetSetting.Min -and [int]$SettingIntValue -le $IntelEthernetSetting.Max)
            {
                if (($SettingIntValue % $IntelEthernetSetting.Step) -eq 0)
                {
                    $bValidIntValue = $true
                }
            }
        }

        if (-not $bValidIntValue)
        {
            $script:ErrorMessagesSet += $Messages.InvalidSettingValue -f $AdapterName, $SettingIntValue, $Setting.RegistryKeyword
        }
        elseif ($SettingIntValue -ne $IntelEthernetSetting.DisplayValue)
        {
            $script:ErrorMessagesSet += $Messages.GenericFailure -f $AdapterName, "Set", $Setting.RegistryKeyword
        }
        else
        {
            return $IntelEthernetSetting
        }
    }
    else
    {
        $bValidEnumValue = $false
        if ($null -ne $DisplayValue)
        {
            if ($IntelEthernetSetting.DescriptionMap -notcontains $DisplayValue -and $IntelEthernetSetting.RegistryKeyword -ne "NetworkAddress")
            {
                $script:ErrorMessagesSet += $Messages.InvalidSettingValue -f $AdapterName, $DisplayValue, $Setting.RegistryKeyword
            }
            elseif ($DisplayValue.ToString() -ne $IntelEthernetSetting.DisplayValue)
            {
                $script:ErrorMessagesSet += $Messages.GenericFailure -f $AdapterName, "Set", $Setting.DisplayName
            }
            else
            {
                $bValidEnumValue = $true
            }
        }
        elseif ($null -ne $RegistryValue)
        {
            if ($IntelEthernetSetting.PossibleValues -notcontains $RegistryValue -and $IntelEthernetSetting.RegistryKeyword -ne "NetworkAddress")
            {
                $script:ErrorMessagesSet += $Messages.InvalidSettingValue -f $AdapterName, $RegistryValue, $Setting.RegistryKeyword
            }
            elseif (($RegistryValue.ToString() -ne $IntelEthernetSetting.RegistryValue))
            {
                $script:ErrorMessagesSet += $Messages.GenericFailure -f $AdapterName, "Set", $Setting.DisplayName
            }
            else
            {
                $bValidEnumValue = $true
            }
        }

        if ($bValidEnumValue)
        {
            return $IntelEthernetSetting
        }
    }
}

function ValidateLAASetting($DisplayValue, $RegistryValue)
{
    $Valid = $False
    do
    {
        if (-not [string]::IsNullOrEmpty($DisplayValue))
        {
            $Value = $DisplayValue
        }
        else
        {
            $Value = $RegistryValue
        }
        # first check if the string has correct length
        if ($Value.Length -eq $NETWORK_ADDRESS_LEN)
        {
            # filter out all non-hexadecimal numbers
            try
            {
                $ValueInt = [Convert]::ToUInt64($Value, 16)
            }
            catch
            {
                break
            }
            # check for reserved and multicast addresses
            if ($ValueInt -band 0x010000000000 -or $ValueInt -eq 0 -or $ValueInt -eq 0xFFFFFFFFFFFF)
            {
                break
            }
            $Valid = $True
        }
    } while($false)

    if ($false -eq $Valid)
    {
        $script:ErrorMessagesSet += $Messages.InvalidSettingValueGeneric
    }
    return $Valid
}

function IsSettingObject($PipedObject)
{
    $bSettingObject = $false
    if ($null -ne $PipedObject -and $null -ne $PipedObject[0].PSTypeNames -and
        ($PipedObject[0].PSTypeNames[0]).Contains("IntelEthernetSetting"))
    {
        $bSettingObject = $true
    }

    return $bSettingObject
}

# SIG # Begin signature block
# MIIorQYJKoZIhvcNAQcCoIIonjCCKJoCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCBSdaeL2QzFYtAd
# pTmlCHumALC4iSA52SaBagifJgWOOaCCEgUwggWeMIIEhqADAgECAhEAzS1l4rws
# CIvYBjRVawV4ujANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQGEwJHQjEbMBkGA1UE
# CBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYDVQQHEwdTYWxmb3JkMRgwFgYDVQQK
# Ew9TZWN0aWdvIExpbWl0ZWQxJDAiBgNVBAMTG1NlY3RpZ28gUlNBIENvZGUgU2ln
# bmluZyBDQTAeFw0yMTA0MDIwMDAwMDBaFw0yMzA0MDIyMzU5NTlaMIGEMQswCQYD
# VQQGEwJVUzETMBEGA1UECAwKQ2FsaWZvcm5pYTEUMBIGA1UEBwwLU2FudGEgQ2xh
# cmExGjAYBgNVBAoMEUludGVsIENvcnBvcmF0aW9uMRIwEAYDVQQLDAlTb2xhcktN
# Q1MxGjAYBgNVBAMMEUludGVsIENvcnBvcmF0aW9uMIIBojANBgkqhkiG9w0BAQEF
# AAOCAY8AMIIBigKCAYEA7CCN9iKpDHOrRceKhlXFP/tf6Lllw2H2fR9KVI4/fQIx
# MU1hXwnlHmAzMCY7IgcCFY4p3F5/MJGKaqYngwOo28Zo6Q1N6ukysA7PSavmF2RY
# WD6VFeya/2H0PoNeRFjHaRzSeynFFeJAFew9r7UReUwM/507sxZYPQuWWIdAEK7H
# Dqp2VlHmgZOXVGHhNO6GFOKpC/C01g6X3x6OquddRNMt5UrZzZzDo5MpJz9SBB2V
# jiqwZ80dvNR2W2xi90cIHh4BkXvB54UNkp4VTVu16T0k3cweo+C39U7GrCAr5Axz
# DETjBvhNtP1sf9SoRV7xY6g5wssfI7yYT9J0gsifn/Vy8MWH355TPoA+PVhbAu0m
# 9FMz4EWu55nnUurNML2jaUxsos21/7ELat12kWC0tq9fhkODjKO8X9PuiBHflZLk
# d3F4QcSMvuGocWGqE77VV3vn8jlvigm2TOV0CfGTQajGMX0jeTRZ19fzBNkt2X9d
# SSGolI/Kj1gSvCggpkUBAgMBAAGjggGQMIIBjDAfBgNVHSMEGDAWgBQO4TqoUzox
# 1Yq+wbutZxoDha00DjAdBgNVHQ4EFgQUshkNuM2SdwJnW4vFy8c4FtUTrbQwDgYD
# VR0PAQH/BAQDAgeAMAwGA1UdEwEB/wQCMAAwEwYDVR0lBAwwCgYIKwYBBQUHAwMw
# EQYJYIZIAYb4QgEBBAQDAgQQMEoGA1UdIARDMEEwNQYMKwYBBAGyMQECAQMCMCUw
# IwYIKwYBBQUHAgEWF2h0dHBzOi8vc2VjdGlnby5jb20vQ1BTMAgGBmeBDAEEATBD
# BgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29S
# U0FDb2RlU2lnbmluZ0NBLmNybDBzBggrBgEFBQcBAQRnMGUwPgYIKwYBBQUHMAKG
# Mmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1JTQUNvZGVTaWduaW5nQ0Eu
# Y3J0MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG
# 9w0BAQsFAAOCAQEAVadLNRW4f/pKMqrbn0BdOoQ8/1EJ87gvVfosei2bLwTEvpmv
# mn2n561H6AFedtIJ6L4FmXII4M4r20i+5LREbI6PpKDmOAf4xW7POxfCRvkTQAZO
# 3zoVxjMQBXo7cZVF1xHCdviXzD1usuIiCF8DLm6z4O/kyeFFNcn816yPQct91Pnk
# SBBVvL+Kwu8xvR+ZIQy632WUA4HnNpRdFnVSzUifEg2GrtsKZR8k+rm2o8K8yjJq
# 3SznwgJQCMVMh3CtRtUwE/c7o/6rvm53fTYJDd3aoPHVgH6S2WqS3+3mQG7A6hTD
# nrP/mYnS4PF7XzxxjZhUlhy4G/MarJPvT9IrNDCCBfUwggPdoAMCAQICEB2iSDBv
# myYY0ILgln0z02owDQYJKoZIhvcNAQEMBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0eTEeMBwGA1UEChMV
# VGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1c3QgUlNBIENl
# cnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTE4MTEwMjAwMDAwMFoXDTMwMTIzMTIz
# NTk1OVowfDELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFuY2hlc3Rl
# cjEQMA4GA1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSQw
# IgYDVQQDExtTZWN0aWdvIFJTQSBDb2RlIFNpZ25pbmcgQ0EwggEiMA0GCSqGSIb3
# DQEBAQUAA4IBDwAwggEKAoIBAQCGIo0yhXoYn0nwli9jCB4t3HyfFM/jJrYlZilA
# hlRGdDFixRDtsocnppnLlTDAVvWkdcapDlBipVGREGrgS2Ku/fD4GKyn/+4uMyD6
# DBmJqGx7rQDDYaHcaWVtH24nlteXUYam9CflfGqLlR5bYNV+1xaSnAAvaPeX7Wpy
# vjg7Y96Pv25MQV0SIAhZ6DnNj9LWzwa0VwW2TqE+V2sfmLzEYtYbC43HZhtKn52B
# xHJAteJf7wtF/6POF6YtVbC3sLxUap28jVZTxvC6eVBJLPcDuf4vZTXyIuosB69G
# 2flGHNyMfHEo8/6nxhTdVZFuihEN3wYklX0Pp6F8OtqGNWHTAgMBAAGjggFkMIIB
# YDAfBgNVHSMEGDAWgBRTeb9aqitKz1SA4dibwJ3ysgNmyzAdBgNVHQ4EFgQUDuE6
# qFM6MdWKvsG7rWcaA4WtNA4wDgYDVR0PAQH/BAQDAgGGMBIGA1UdEwEB/wQIMAYB
# Af8CAQAwHQYDVR0lBBYwFAYIKwYBBQUHAwMGCCsGAQUFBwMIMBEGA1UdIAQKMAgw
# BgYEVR0gADBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8vY3JsLnVzZXJ0cnVzdC5j
# b20vVVNFUlRydXN0UlNBQ2VydGlmaWNhdGlvbkF1dGhvcml0eS5jcmwwdgYIKwYB
# BQUHAQEEajBoMD8GCCsGAQUFBzAChjNodHRwOi8vY3J0LnVzZXJ0cnVzdC5jb20v
# VVNFUlRydXN0UlNBQWRkVHJ1c3RDQS5jcnQwJQYIKwYBBQUHMAGGGWh0dHA6Ly9v
# Y3NwLnVzZXJ0cnVzdC5jb20wDQYJKoZIhvcNAQEMBQADggIBAE1jUO1HNEphpNve
# aiqMm/EAAB4dYns61zLC9rPgY7P7YQCImhttEAcET7646ol4IusPRuzzRl5ARokS
# 9At3WpwqQTr81vTr5/cVlTPDoYMot94v5JT3hTODLUpASL+awk9KsY8k9LOBN9O3
# ZLCmI2pZaFJCX/8E6+F0ZXkI9amT3mtxQJmWunjxucjiwwgWsatjWsgVgG10Xkp1
# fqW4w2y1z99KeYdcx0BNYzX2MNPPtQoOCwR/oEuuu6Ol0IQAkz5TXTSlADVpbL6f
# ICUQDRn7UJBhvjmPeo5N9p8OHv4HURJmgyYZSJXOSsnBf/M6BZv5b9+If8AjntIe
# Q3pFMcGcTanwWbJZGehqjSkEAnd8S0vNcL46slVaeD68u28DECV3FTSK+TbMQ5Lk
# uk/xYpMoJVcp+1EZx6ElQGqEV8aynbG8HArafGd+fS7pKEwYfsR7MUFxmksp7As9
# V1DSyt39ngVR5UR43QHesXWYDVQk/fBO4+L4g71yuss9Ou7wXheSaG3IYfmm8SoK
# C6W59J7umDIFhZ7r+YMp08Ysfb06dy6LN0KgaoLtO0qqlBCk4Q34F8W2WnkzGJLj
# tXX4oemOCiUe5B7xn1qHI/+fpFGe+zmAEc3btcSnqIBv5VPU4OOiwtJbGvoyJi1q
# V3AcPKRYLqPzW0sH3DJZ84enGm1YMIIGZjCCBE6gAwIBAgITMwAAAES3P/zvWs+i
# egAAAAAARDANBgkqhkiG9w0BAQUFADB/MQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMSkwJwYDVQQDEyBNaWNyb3NvZnQgQ29kZSBWZXJpZmljYXRp
# b24gUm9vdDAeFw0xNTA3MjIyMTAzNDlaFw0yNTA3MjIyMTAzNDlaMIGIMQswCQYD
# VQQGEwJVUzETMBEGA1UECBMKTmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENp
# dHkxHjAcBgNVBAoTFVRoZSBVU0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNF
# UlRydXN0IFJTQSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eTCCAiIwDQYJKoZIhvcN
# AQEBBQADggIPADCCAgoCggIBAIASZRc2DsPbCLPQrFcNdu3NJ9NMrVCDYeKqIE0J
# LWQJ3M6Jn8w9qez2z8Hc8dOx1ns3KBErR9o5xrw6GbRfpr19naNjQrZ28qk7K5H4
# 4m/Q7BYgkAk+4uh0yRi0kdRiZNt/owbxiBhqkCI8vP4T8IcUe/bkH47U5FHGEWdG
# CFHLhhRUP7wz/n5snP8WnRi9UY41pqdmyHJn2yFmsdSbeAPAUDrozPDcvJ5M/q8F
# ljUfV1q3/875PbcstvZU3cjnEjpNrkyKt1yatLcgPcp/IjSufjtoZgFE5wFORlOb
# M2D3lL5TN5BzQ/Myw1Pv26r+dE5px2uMYJPexMcM3+EyrsyTO1F4lWeL7j1W/gzQ
# aQ8bD/MlJmszbfduR/pzQ+V+DqVmsSl8MoRjVYnEDcGTVDAZE6zTfTen6106bDVc
# 20HXEtqpSQvf2ICKCZNijrVmzyWIzYS4sT+kOQ/ZAp7rEkyVfPNrBaleFoPMuGfi
# 6BOdzFuC00yz7Vv/3uVzrCM7LQC/NVV0CUnYSVgaf5I25lGSDvMmfRxNF7zJ7EMm
# 0L9BX0CpRET0medXh55QH1dUqD79dGMvsVBlCeZYQi5DGky08CVHWfoEHpPUJkZK
# UIGy3r54t/xnFeHJV4QeD2PW6WK61l9VLupcxigIBCU5uA4rqfJMlxwHPw1S9e3v
# L4IPAgMBAAGjgdAwgc0wEwYDVR0lBAwwCgYIKwYBBQUHAwMwEgYDVR0TAQH/BAgw
# BgEB/wIBAjAdBgNVHQ4EFgQUU3m/WqorSs9UgOHYm8Cd8rIDZsswCwYDVR0PBAQD
# AgGGMB8GA1UdIwQYMBaAFGL7CiFbf0NuEdoJVFBr9dKWcfGeMFUGA1UdHwROMEww
# SqBIoEaGRGh0dHA6Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3Rz
# L01pY3Jvc29mdENvZGVWZXJpZlJvb3QuY3JsMA0GCSqGSIb3DQEBBQUAA4ICAQBr
# IpM8PTlUcWRrDvLkPDARxSBKS4YPkvH/M3k62eSYpw5AoCKAfmGy4KcZzyaVMSpl
# 1GpPMYbqwMYuxWSMPUhZzQsvdD2UJhMQQtSXmCdePHbSeGkdGmTnBXJ14OtmQEOf
# jwxG/5dgpshnrRAIm2Km6b46itMHTZ9ykyW8BhHgLJA4Pmcc/RnXnpDOPcLg52Gs
# wOUE9R6ZVAyRDQFWcTeuJ9SeQyKlySfNTeVxEjkkpUFWh/+8VRQPJcqJ7seX5dIT
# /z1+GqCPP8gs16Nw0MdgwPzYPlHnl8Y+O+3PeL6KyuPE8qen7Z6uCAKPoFLbch7V
# O8NNn476m3DH+OO/bD+Sm+Q3PuxqjCn5waK/iz4aaWb7HGNPJgHJAsQ+0v/DQ6gb
# /Zn61LylueKTLzsBxdH0Oi9ow+Bkt1qVXkbMB4NpuzwFklZzNXNFmE582BKlt0Lp
# omP2QmAYcNE7bzHAh8fmceHzRhbp9bhys+ltH2ImSaNJi91ox4toVvfe/PqHJLgD
# gReP5fFnah2u03T3jKVdswuOQimWzknEd35mfAEXGmwUJMOwF3cF2BpAt4Zr2OR7
# QKx+305vJPkggIKMM+fl+inYndqLcF0ryR2CTAtny4RBnucGfhGDRC2KGe70f5rd
# eRw3GR6fP4wpug1cEIY3bEjNRV3NcLy80U1d2MW4djGCFf4wghX6AgEBMIGRMHwx
# CzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNV
# BAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEkMCIGA1UEAxMb
# U2VjdGlnbyBSU0EgQ29kZSBTaWduaW5nIENBAhEAzS1l4rwsCIvYBjRVawV4ujAN
# BglghkgBZQMEAgEFAKBqMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisG
# AQQBgjcCAQsxDjAMBgorBgEEAYI3AgEWMC8GCSqGSIb3DQEJBDEiBCA/UxG70MiU
# rh3VI9sbed4Jryduow2wqFmlBHcyZGx5WjANBgkqhkiG9w0BAQEFAASCAYBcB423
# W3dXty+VyzAMyV55CsB7Vd5lrl325N0y8kEtpJf9vHIbQ+CTXvPA/3Yl/qJL7t+T
# T+bUR+TgwhvRWHM/RgxUslNybzGUmvmtk7Y7OH9v7NX4vaT4vkgS9SHaInVDLGjn
# kcjn4t1byzxsatU2J78H5rKXIDhb5R360yXpum4cWahqyFQdq/GylIE+1687fkYm
# EGuJtv0BPbTCCIBd1IO0zSsgptyigzz80ghf+0N6MqjhNjnHbZgqsv0MkoZOvNes
# n3z7M/q5uCTmelF/BDpK2ya1Cpwk9+elOLY7pdjaXdy1GFx1Rc9stDhpRpaUpDb5
# xfqXyfPAZICwQMQ6LM0vO7iB06CUGdhYihRTc2IzC6jjBiChoa/6PfaURUpVk7XY
# hCPblGhnUdTGVwm+dL4JoDUjkHBKiw2y2RWwerkxFw044Kp4Go0+c4+BwfIqLivQ
# PS5R0IFt1NAGsvV16wN1lz3OsqAfgeAZUXquWWXfnJyq/j8Laq6VbwwHgt6hghNR
# MIITTQYKKwYBBAGCNwMDATGCEz0wghM5BgkqhkiG9w0BBwKgghMqMIITJgIBAzEP
# MA0GCWCGSAFlAwQCAgUAMIHwBgsqhkiG9w0BCRABBKCB4ASB3TCB2gIBAQYKKwYB
# BAGyMQIBATAxMA0GCWCGSAFlAwQCAQUABCDsRNIFdQjeqJOlzWyYCSL7Turs5J0p
# 3bkiJWiCjr1KdwIVAK0j1y4wag5ZdNdGFoBo32wPpLAIGA8yMDIzMDExMDIzMDA0
# MVqgbqRsMGoxCzAJBgNVBAYTAkdCMRMwEQYDVQQIEwpNYW5jaGVzdGVyMRgwFgYD
# VQQKEw9TZWN0aWdvIExpbWl0ZWQxLDAqBgNVBAMMI1NlY3RpZ28gUlNBIFRpbWUg
# U3RhbXBpbmcgU2lnbmVyICMzoIIN6jCCBvYwggTeoAMCAQICEQCQOX+a0ko6E/K9
# kV8IOKlDMA0GCSqGSIb3DQEBDAUAMH0xCzAJBgNVBAYTAkdCMRswGQYDVQQIExJH
# cmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcTB1NhbGZvcmQxGDAWBgNVBAoTD1Nl
# Y3RpZ28gTGltaXRlZDElMCMGA1UEAxMcU2VjdGlnbyBSU0EgVGltZSBTdGFtcGlu
# ZyBDQTAeFw0yMjA1MTEwMDAwMDBaFw0zMzA4MTAyMzU5NTlaMGoxCzAJBgNVBAYT
# AkdCMRMwEQYDVQQIEwpNYW5jaGVzdGVyMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0
# ZWQxLDAqBgNVBAMMI1NlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcgU2lnbmVyICMz
# MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAkLJxP3nh1LmKF8zDl8KQ
# lHLtWjpvAUN/c1oonyR8oDVABvqUrwqhg7YT5EsVBl5qiiA0cXu7Ja0/WwqkHy9s
# fS5hUdCMWTc+pl3xHl2AttgfYOPNEmqIH8b+GMuTQ1Z6x84D1gBkKFYisUsZ0vCW
# yUQfOV2csJbtWkmNfnLkQ2t/yaA/bEqt1QBPvQq4g8W9mCwHdgFwRd7D8EJp6v8m
# zANEHxYo4Wp0tpxF+rY6zpTRH72MZar9/MM86A2cOGbV/H0em1mMkVpCV1VQFg1L
# dHLuoCox/CYCNPlkG1n94zrU6LhBKXQBPw3gE3crETz7Pc3Q5+GXW1X3KgNt1c1i
# 2s6cHvzqcH3mfUtozlopYdOgXCWzpSdoo1j99S1ryl9kx2soDNqseEHeku8Pxeyr
# 3y1vGlRRbDOzjVlg59/oFyKjeUFiz/x785LaruA8Tw9azG7fH7wir7c4EJo0pwv/
# /h1epPPuFjgrP6x2lEGdZB36gP0A4f74OtTDXrtpTXKZ5fEyLVH6Ya1N6iaObfyp
# SJg+8kYNabG3bvQF20EFxhjAUOT4rf6sY2FHkbxGtUZTbMX04YYnk4Q5bHXgHQx6
# WYsuy/RkLEJH9FRYhTflx2mn0iWLlr/GreC9sTf3H99Ce6rrHOnrPVrd+NKQ1Uma
# Oh2DGld/HAHCzhx9zPuWFcUCAwEAAaOCAYIwggF+MB8GA1UdIwQYMBaAFBqh+GEZ
# IA/DQXdFKI7RNV8GEgRVMB0GA1UdDgQWBBQlLmg8a5orJBSpH6LfJjrPFKbx4DAO
# BgNVHQ8BAf8EBAMCBsAwDAYDVR0TAQH/BAIwADAWBgNVHSUBAf8EDDAKBggrBgEF
# BQcDCDBKBgNVHSAEQzBBMDUGDCsGAQQBsjEBAgEDCDAlMCMGCCsGAQUFBwIBFhdo
# dHRwczovL3NlY3RpZ28uY29tL0NQUzAIBgZngQwBBAIwRAYDVR0fBD0wOzA5oDeg
# NYYzaHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdvUlNBVGltZVN0YW1waW5n
# Q0EuY3JsMHQGCCsGAQUFBwEBBGgwZjA/BggrBgEFBQcwAoYzaHR0cDovL2NydC5z
# ZWN0aWdvLmNvbS9TZWN0aWdvUlNBVGltZVN0YW1waW5nQ0EuY3J0MCMGCCsGAQUF
# BzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG9w0BAQwFAAOCAgEA
# c9rtaHLLwrlAoTG7tAOjLRR7JOe0WxV9qOn9rdGSDXw9NqBp2fOaMNqsadZ0VyQ/
# fg882fXDeSVsJuiNaJPO8XeJOX+oBAXaNMMU6p8IVKv/xH6WbCvTlOu0bOBFTSyy
# 9zs7WrXB+9eJdW2YcnL29wco89Oy0OsZvhUseO/NRaAA5PgEdrtXxZC+d1SQdJ4L
# T03EqhOPl68BNSvLmxF46fL5iQQ8TuOCEmLrtEQMdUHCDzS4iJ3IIvETatsYL254
# rcQFtOiECJMH+X2D/miYNOR35bHOjJRs2wNtKAVHfpsu8GT726QDMRB8Gvs8GYDR
# C3C5VV9HvjlkzrfaI1Qy40ayMtjSKYbJFV2Ala8C+7TRLp04fDXgDxztG0dInCJq
# VYLZ8roIZQPl8SnzSIoJAUymefKithqZlOuXKOG+fRuhfO1WgKb0IjOQ5IRT/Cr6
# wKeXqOq1jXrO5OBLoTOrC3ag1WkWt45mv1/6H8Sof6ehSBSRDYL8vU2Z7cnmbDb+
# d0OZuGktfGEv7aOwSf5bvmkkkf+T/FdpkkvZBT9thnLTotDAZNI6QsEaA/vQ7Zoh
# uD+vprJRVNVMxcofEo1XxjntXP/snyZ2rWRmZ+iqMODSrbd9sWpBJ24DiqN04IoJ
# gm6/4/a3vJ4LKRhogaGcP24WWUsUCQma5q6/YBXdhvUwggbsMIIE1KADAgECAhAw
# D2+s3WaYdHypRjaneC25MA0GCSqGSIb3DQEBDAUAMIGIMQswCQYDVQQGEwJVUzET
# MBEGA1UECBMKTmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENpdHkxHjAcBgNV
# BAoTFVRoZSBVU0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNFUlRydXN0IFJT
# QSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eTAeFw0xOTA1MDIwMDAwMDBaFw0zODAx
# MTgyMzU5NTlaMH0xCzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNo
# ZXN0ZXIxEDAOBgNVBAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRl
# ZDElMCMGA1UEAxMcU2VjdGlnbyBSU0EgVGltZSBTdGFtcGluZyBDQTCCAiIwDQYJ
# KoZIhvcNAQEBBQADggIPADCCAgoCggIBAMgbAa/ZLH6ImX0BmD8gkL2cgCFUk7nP
# oD5T77NawHbWGgSlzkeDtevEzEk0y/NFZbn5p2QWJgn71TJSeS7JY8ITm7aGPwEF
# kmZvIavVcRB5h/RGKs3EWsnb111JTXJWD9zJ41OYOioe/M5YSdO/8zm7uaQjQqzQ
# FcN/nqJc1zjxFrJw06PE37PFcqwuCnf8DZRSt/wflXMkPQEovA8NT7ORAY5unSd1
# VdEXOzQhe5cBlK9/gM/REQpXhMl/VuC9RpyCvpSdv7QgsGB+uE31DT/b0OqFjIpW
# cdEtlEzIjDzTFKKcvSb/01Mgx2Bpm1gKVPQF5/0xrPnIhRfHuCkZpCkvRuPd25Ff
# nz82Pg4wZytGtzWvlr7aTGDMqLufDRTUGMQwmHSCIc9iVrUhcxIe/arKCFiHd6QV
# 6xlV/9A5VC0m7kUaOm/N14Tw1/AoxU9kgwLU++Le8bwCKPRt2ieKBtKWh97oaw7w
# W33pdmmTIBxKlyx3GSuTlZicl57rjsF4VsZEJd8GEpoGLZ8DXv2DolNnyrH6jaFk
# yYiSWcuoRsDJ8qb/fVfbEnb6ikEk1Bv8cqUUotStQxykSYtBORQDHin6G6UirqXD
# TYLQjdprt9v3GEBXc/Bxo/tKfUU2wfeNgvq5yQ1TgH36tjlYMu9vGFCJ10+dM70a
# tZ2h3pVBeqeDAgMBAAGjggFaMIIBVjAfBgNVHSMEGDAWgBRTeb9aqitKz1SA4dib
# wJ3ysgNmyzAdBgNVHQ4EFgQUGqH4YRkgD8NBd0UojtE1XwYSBFUwDgYDVR0PAQH/
# BAQDAgGGMBIGA1UdEwEB/wQIMAYBAf8CAQAwEwYDVR0lBAwwCgYIKwYBBQUHAwgw
# EQYDVR0gBAowCDAGBgRVHSAAMFAGA1UdHwRJMEcwRaBDoEGGP2h0dHA6Ly9jcmwu
# dXNlcnRydXN0LmNvbS9VU0VSVHJ1c3RSU0FDZXJ0aWZpY2F0aW9uQXV0aG9yaXR5
# LmNybDB2BggrBgEFBQcBAQRqMGgwPwYIKwYBBQUHMAKGM2h0dHA6Ly9jcnQudXNl
# cnRydXN0LmNvbS9VU0VSVHJ1c3RSU0FBZGRUcnVzdENBLmNydDAlBggrBgEFBQcw
# AYYZaHR0cDovL29jc3AudXNlcnRydXN0LmNvbTANBgkqhkiG9w0BAQwFAAOCAgEA
# bVSBpTNdFuG1U4GRdd8DejILLSWEEbKw2yp9KgX1vDsn9FqguUlZkClsYcu1UNvi
# ffmfAO9Aw63T4uRW+VhBz/FC5RB9/7B0H4/GXAn5M17qoBwmWFzztBEP1dXD4rzV
# WHi/SHbhRGdtj7BDEA+N5Pk4Yr8TAcWFo0zFzLJTMJWk1vSWVgi4zVx/AZa+clJq
# O0I3fBZ4OZOTlJux3LJtQW1nzclvkD1/RXLBGyPWwlWEZuSzxWYG9vPWS16toytC
# iiGS/qhvWiVwYoFzY16gu9jc10rTPa+DBjgSHSSHLeT8AtY+dwS8BDa153fLnC6N
# Ixi5o8JHHfBd1qFzVwVomqfJN2Udvuq82EKDQwWli6YJ/9GhlKZOqj0J9QVst9Jk
# WtgqIsJLnfE5XkzeSD2bNJaaCV+O/fexUpHOP4n2HKG1qXUfcb9bQ11lPVCBbqvw
# 0NP8srMftpmWJvQ8eYtcZMzN7iea5aDADHKHwW5NWtMe6vBE5jJvHOsXTpTDeGUg
# Ow9Bqh/poUGd/rG4oGUqNODeqPk85sEwu8CgYyz8XBYAqNDEf+oRnR4GxqZtMl20
# OAkrSQeq/eww2vGnL8+3/frQo4TZJ577AWZ3uVYQ4SBuxq6x+ba6yDVdM3aO8Xwg
# DCp3rrWiAoa6Ke60WgCxjKvj+QrJVF3UuWp0nr1IrpgxggQtMIIEKQIBATCBkjB9
# MQswCQYDVQQGEwJHQjEbMBkGA1UECBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYD
# VQQHEwdTYWxmb3JkMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxJTAjBgNVBAMT
# HFNlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcgQ0ECEQCQOX+a0ko6E/K9kV8IOKlD
# MA0GCWCGSAFlAwQCAgUAoIIBazAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQAQQw
# HAYJKoZIhvcNAQkFMQ8XDTIzMDExMDIzMDA0MVowPwYJKoZIhvcNAQkEMTIEMFlU
# EOzS8jaV0g79rgwTsYRV3RNODIHNF8fiyLq4ld48aDQvGv5lXn0CMFQkTR6BJjCB
# 7QYLKoZIhvcNAQkQAgwxgd0wgdowgdcwFgQUqzQBOqxAlzGfCBrwsxjhg/gPeIEw
# gbwEFALWW5Xig3DBVwCV+oj5I92Tf62PMIGjMIGOpIGLMIGIMQswCQYDVQQGEwJV
# UzETMBEGA1UECBMKTmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENpdHkxHjAc
# BgNVBAoTFVRoZSBVU0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNFUlRydXN0
# IFJTQSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eQIQMA9vrN1mmHR8qUY2p3gtuTAN
# BgkqhkiG9w0BAQEFAASCAgAlW6nWgrVLbavkUHK1vYA+B4zk5jAz2uZpG5Rb3teP
# Pab1SsLwS11zmpJHi8mjeEBMsXGJ+4ROyCz0XsYsEqMEqYF7y6mZ//vHLuGCJ2bR
# 4FVKDQ5eLx2FmFFZSNUUzpQWuwprHApv+5e/vLcaJ8vjK7aL34f70oI5iOTiwRAc
# lt0dAoEnFk2z4C9mHIqzO8fbBauYzPjUdBbl2k5QNnzwtMLdCaG5LY5hQmF7aIx7
# WzKCdyrCYm11A+M0wnFjpy0l9rnY48atbf/Wl+p4xsUEfXaJ5mE8RQ4xXQHRDtHD
# pyvoRTPnDGTid5efDK/B/xQV14xjPFQVkQ+goi6ikC+naYDcYeYmdUCnNWXoWY8S
# eNJxSqt0MXxiWiZQxj3IrmOK3IKbzA79V3xqVPS7hLfjR7+Tgno7xK9oiQcGpKXS
# ZM+6R+eM1q3mcRK4EEMH/5VpZB5tltaAO1cY7gwwQStpb6gkJlsioK+yfhpbaJM+
# FIwTMOZrDxd5sHmmPd7VraF1EuuKs+7CG+ea/wB8XuO45gP0uVjA3udj44Zsb+nw
# vuQ2kXyW3APtRhgQWGpFMUmXNjsXdesDuUfsKEvo06ie/iqlEp0yWnGtgHwZ+fv5
# ZCQrttFG0nLbDt/ZRGgfE6e/NIr492aZdFSw/VQEbghgyB4NkKayMBEgyI0jKHzO
# Mg==
# SIG # End signature block
