# Below setting allows to run local scripts without digital signature
Set-ExecutionPolicy -ExecutionPolicy RemoteSigned -Scope Process -Force

pnputil /scan-devices

$script_root = Split-Path -Parent $MyInvocation.MyCommand.Path
$driver_inf_path = Join-Path -Path $script_root -ChildPath "..\drivers\x64\npu.inf"
$driver_extn_inf_path= Join-Path -Path $script_root -ChildPath "..\drivers\x64\npu_extension_internal.inf"

$global:use_extn_inf = $true
$global:disable_non_MEP = $true

# function to query the device instance ID
function Get-InstanceId {
	param( [string]$device_id )
	
	$instance_id = ""
	$pnputil_output = pnputil /enum-devices /connected | Select-String -Pattern "pci" | Select-String -Pattern $device_id

	foreach ($line in $pnputil_output) {
		$tokens = $line -split ":"
		$instance_id = $tokens[1].Trim()
		$instance_id = $instance_id -replace " "
	}

	return $instance_id	
}

# function to verify if device exists on the machine
function  Get-DeviceInstanceId {
	
	$device_instance_id = ""
	
	if ($device_instance_id = Get-InstanceId -device_id "DEV_6240") {
		Write-Host "Device Instance ID: $device_instance_id"
		$global:use_extn_inf=$false
		$global:disable_non_MEP=$false
		return $device_instance_id
	}
	else {
		$device_ids = "DEV_AD1D", "DEV_7D1D", "DEV_643E", "DEV_B03E", "DEV_FD3E"
		
		foreach ($device_id in $device_ids) {	
            		$device_instance_id = Get-InstanceId -device_id $device_id
			
            		if($device_instance_id -ne "") {         
				Write-Host "device Instance ID: $device_instance_id"
				return $device_instance_id
			}
		}
	}
	
	if($device_instance_id -eq "") {
		Write-Host "Device not found"
		exit 1
	}
}

# function to query the status of device
function Get-DeviceStatus {

	$status = ""
	$device_instance_id = Get-DeviceInstanceId 
	$pnputil_output = pnputil /enum-devices /instanceid "$device_instance_id" | Select-String -Pattern "Status:"
	
	foreach ( $line in $pnputil_output ) {
		$tokens = $line -split ":"
		$status = $tokens[1].Trim()
		$status = $status -replace " "
	}
	
	if( $status -eq "Problem" ) {
		Write-Host "Device status: Problem"
		exit 1
	} 
	elseif ( $status -eq "Started" ) {
		Write-Host "Device status: Driver installed"
	} 
	elseif ( $status -eq "Stopped" ) {
		Write-Host "Device status: Driver not installed"
	} 
	elseif ( $status -eq "Disabled" ) {
		Write-Host "Device status: Disabled"
	}  
	else {
		Write-Host "Device status: Unknown"
	}
	
	$result = [PSCustomObject]@{
		status = $status
		device_instance_id = $device_instance_id
	}
	
	return $result
}

# function to query a driver name
function Get-DriverName {	
	param ([string]$device_instance_id)

	$pnputilOutput = pnputil /enum-devices /instanceid "$device_instance_id" | Select-String -Pattern "Driver Name"
	$names = @()

	foreach ($line in $pnputilOutput) {
		$tokens = $line -split ":"
		$name = $tokens[1].Trim()
		$name = $name -replace " "
		$names += $name
	}
	
	return $names
}

#function to cleanup dangling npu and npu_extension infs
function Clear-NpuInf {
	
	$enumDriversOutput = pnputil.exe /enum-drivers

	$originalNamePattern = 'Original Name:\s+(npu.*)'
	$publishedNamePattern = 'Published Name:\s+(.+)'

	$matches = $enumDriversOutput | Select-String -Pattern $originalNamePattern -Context 1

	foreach ($match in $matches) {
		$previousLine = $match.Context.PreContext[0]
		$publishedName = ($previousLine | Select-String -Pattern $publishedNamePattern).Matches.Groups[1].Value

		Write-Host "Deleting: $publishedName"
		pnputil /delete-driver $publishedName /uninstall /force
	}
	
	return
}

# function to add EnableNonMep regkey 
function Add-NonMep-Regkey {

	$intel_key_path = "HKLM:\SOFTWARE\Intel"
	$vpu_key_path = "HKLM:\SOFTWARE\Intel\VPU"

	# Check if each key exists, and create them if they don't
	if (-not (Test-Path $intel_key_path)) {
		Write-Host "Creating registry key: $intel_key_path"
		New-Item -Path $intel_key_path -Force
	}

	if (-not (Test-Path $vpu_key_path)) {
		Write-Host "Creating registry key: $vpu_key_path"
		New-Item -Path $vpu_key_path -Force
	}
	
	Set-ItemProperty -Path $vpu_key_path -Name "EnableNonMep" -Value 1 -Type "DWord"
}

# function to install a driver
function Install-Driver {
	param ([string]$driver_inf_path)
	
	Write-Host "Installing driver."
	
	if( $global:disable_non_MEP -eq $false ) {
		Add-NonMep-Regkey
	}
	
	pnputil /add-driver $driver_inf_path /install
	return
}

# function to install extension inf
function Install-ExtensionInf {
	param ([string]$driver_extn_inf_path)
	
	if( $global:use_extn_inf -ne $false) {
		Write-Host "Installing extension inf."
		pnputil /add-driver $driver_extn_inf_path /install
	}
	
	return
}

# function to uninstall driver
function Uninstall-Driver {
	param ([string]$device_instance_id)

	$driver_names = Get-DriverName -device_instance_id $device_instance_id
	
	foreach ($driver_name in $driver_names) {
		if ( $driver_name -ne "" ) {			
            		Write-Host "Uninstalling driver : $driver_name"
			pnputil /delete-driver $driver_name /uninstall /force

			if( $global:disable_non_MEP -eq $false ) {
                		Remove-Item -Path "HKLM:\SOFTWARE\SOFTWARE\Intel\VPU\EnableNonMep" -ErrorAction SilentlyContinue
			}
		} 
	}
	
	Clear-NpuInf
	return
}

# function to enable device
function Enable-Device {
	param ([string]$device_instance_id)
	
	Write-Host "Enabling device"
	
	if( $global:disable_non_MEP -eq $false ) {
		Add-NonMep-Regkey
	}
	
	pnputil /enable-device /instanceid $device_instance_id
	
	return
}

# function to disable device
function Disable-Device {
	param ([string]$device_instance_id)
	
	Write-Host "Disabling device"
	
	pnputil /disable-device /instanceid $device_instance_id

	if( $global:disable_non_MEP -eq $false ) {
		Remove-Item -Path "HKLM:\SOFTWARE\SOFTWARE\Intel\VPU\EnableNonMep" -ErrorAction SilentlyContinue
	}
	
	return
}

$result = Get-DeviceStatus
$device_status = $result.status
$device_instance_id = $result.device_instance_id

if ( $args[0] -eq "install_driver" ) {
	Uninstall-Driver -device_instance_id $device_instance_id
	Install-Driver $driver_inf_path
	Install-ExtensionInf $driver_extn_inf_path
	
	if ( $device_status -eq "Disabled" ) {
		Enable-Device -device_instance_id $device_instance_id
	}
}
elseif ( $args[0] -eq "uninstall_driver" ) {
	Uninstall-Driver -device_instance_id $device_instance_id
}
elseif ( $args[0] -eq "enable_device" ) {
	if ( $device_status -ne "Enabled" )	{
		Enable-Device -device_instance_id $device_instance_id
	}
}
elseif ( $args[0] -eq "disable_device" ) {
	if ( $device_status -ne "Disabled" ) {
		Disable-Device -device_instance_id $device_instance_id
	}
}
elseif ( $args[0] -eq "clean" ) {
	Clear-NpuInf
}
else {
	Write-Host "Invalid operation specified. Valid operations are: `n1.install_driver`n2.uninstall_driver`n3.enable_device`n4.disable_device`n5.clean "
	exit 1
}

Get-DeviceStatus
