# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2022 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSAvoidUsingPositionalParameters", "", Scope="function")]
[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSUseShouldProcessForStateChangingFunctions", "", Scope="function")] Param()

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Get-IntelEthernetSetting
{
    [CmdletBinding()]
    Param(
    [parameter(Mandatory = $false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $Name = '',
    [parameter(Mandatory = $false, Position=0, ValueFromPipeline = $true)]
    [ValidateNotNullOrEmpty()]
    [object[]]
    $Adapter = $null,
    [parameter(Mandatory = $false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $DisplayName = '',
    [parameter(Mandatory = $false)]
    [SupportsWildcards()]
    [ValidateNotNullOrEmpty()]
    [String[]]
    $RegistryKeyword = ''
    )
    Begin
    {
        $AdapterName = $Name
        $script:ErrorMessagesGet = @()
        $script:WarningMessagesGet = @()
        $FinalObject = @()
        GetIntelEthernetDevices
        $script:MSNetAdvProperty = Get-NetAdapterAdvancedProperty -ErrorAction SilentlyContinue
    }
    Process
    {
        $Adapters = $Adapter
        $PreProcessedAdapterNames = ValidateGetAdapterNameParams $AdapterName $Adapters
        $AdapterNamesArray = @(GetSupportedAdapters $PreProcessedAdapterNames)
        $AdapterSettings = @()
        do
        {
            if (!(ValidateSettingParams $DisplayName $RegistryKeyword))
            {
                break
            }

            foreach ($a in $AdapterNamesArray)
            {
                if ([string]::IsNullOrEmpty($DisplayName) -and [string]::IsNullOrEmpty($RegistryKeyword))
                {
                    $AdapterSettings = GetAllAdapterSettings $a
                }
                else
                {
                    $AdapterSettings = GetSettings $DisplayName $RegistryKeyword $a
                }

                foreach ($Setting in $AdapterSettings)
                {
                    switch ([int]($Setting.DisplayParameterType))
                    {
                        ([int][DisplayParameterType]::int)   {$FinalObject += GatherIntOutput $a $Setting; break}
                        ([int][DisplayParameterType]::long)  {$FinalObject += GatherIntOutput $a $Setting; break}
                        ([int][DisplayParameterType]::word)  {$FinalObject += GatherIntOutput $a $Setting; break}
                        ([int][DisplayParameterType]::dword) {$FinalObject += GatherIntOutput $a $Setting; break}
                        ([int][DisplayParameterType]::enum)  {$FinalObject += GatherEnumOutput $a $Setting; break}
                        ([int][DisplayParameterType]::edit)  {$FinalObject += GatherEnumOutput $a $Setting; break}
                        default {$FinalObject += GatherEnumOutput $a $Setting; break}
                    }
                }

            }
        } while ($false)
    }
    End
    {
        $FinalObject

        foreach ($WarningMessage in $WarningMessagesGet)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $ErrorMessagesGet)
        {
            Write-Error $ErrorMessage
        }
    }
}

function GetAllAdapterSettings($AdapterName)
{
    return $MSNetAdvProperty | Where-Object {$_.InterfaceDescription -eq $AdapterName}
}

function GetDisplayNameAdapterSettings($AdapterName, $SettingNameArray)
{
    foreach ($SettingName in $SettingNameArray)
    {
        $AdapterSettings += $MSNetAdvProperty | Where-Object {$_.InterfaceDescription -eq $AdapterName -and $_.DisplayName -eq $SettingName}
    }

    return $AdapterSettings
}

function GetRegistryKeywordAdapterSettings($AdapterName, $SettingNameArray)
{
    foreach ($SettingName in $SettingNameArray)
    {
        $AdapterSettings += $MSNetAdvProperty | Where-Object {$_.InterfaceDescription -eq $AdapterName -and $_.RegistryKeyword -eq $SettingName}
    }

    return $AdapterSettings
}

function ValidateSettingParams($DisplayName, $RegistryKeyword)
{
    $result = $true

    if (-not [string]::IsNullOrEmpty($DisplayName) -and -not [string]::IsNullOrEmpty($RegistryKeyword))
    {
        $script:ErrorMessagesGet += $Messages.InvalidParams
        $result = $false
    }

    return $result
}

function GetSettings($DisplayName, $RegistryKeyword, $AdapterName)
{
    $SettingArray = @()

    if (-not [string]::IsNullOrEmpty($DisplayName))
    {
        foreach ($TmpDisplayName in $DisplayName)
        {
            $TmpSetting = $MSNetAdvProperty | Where-Object {$_.InterfaceDescription -eq $AdapterName -and $_.DisplayName -like $TmpDisplayName}
            if ($null -eq $TmpSetting)
            {
                $script:WarningMessagesGet += $Messages.InvalidSetting -f $AdapterName, $TmpDisplayName
            }
            else
            {
                $SettingArray += $TmpSetting
            }
        }
    }

    if (-not [string]::IsNullOrEmpty($RegistryKeyword))
    {
        foreach ($TmpRegistryKeyword in $RegistryKeyword)
        {
            $TmpSetting = $MSNetAdvProperty | Where-Object {$_.InterfaceDescription -eq $AdapterName -and $_.RegistryKeyword -like $TmpRegistryKeyword}
            if ($null -eq $TmpSetting)
            {
                $script:WarningMessagesGet += $Messages.InvalidSetting -f $AdapterName, $TmpRegistryKeyword
            }
            else
            {
                $SettingArray += $TmpSetting
            }
        }
    }

    return $SettingArray
}

function GatherEnumOutput($AdapterName, $Setting)
{
    $SettingMiniHelp = GetSettingMinihelp($Setting)

    # Assemble it all together in PSCustomObject
    return [PsCustomObject] @{
        PSTypeName      = 'IntelEthernetSettingEnum'
        Caption         = $Setting.RegistryKeyword
        CurrentValue    = $Setting.RegistryValue
        DefaultValue    = $Setting.DefaultRegistryValue
        Description     = $Setting.DisplayName
        DescriptionMap  = $Setting.ValidDisplayValues
        DisplayName     = $Setting.DisplayName
        DisplayValue    = $Setting.DisplayValue
        Minihelp        = $SettingMiniHelp
        Name            = $AdapterName
        ParentId        = $Setting.InstanceID.split(':')[0]
        PossibleValues  = $Setting.ValidRegistryValues
        RegistryKeyword = $Setting.RegistryKeyword
        RegistryValue   = $Setting.RegistryValue
        }
}

function GatherIntOutput($AdapterName, $Setting)
{
    $SettingMiniHelp = GetSettingMinihelp($Setting)

    # Assemble it all together in PSCustomObject
    return [PsCustomObject] @{
        PSTypeName      = 'IntelEthernetSettingInt'
        Base            = $Setting.NumericParameterBaseValue
        Caption         = $Setting.RegistryKeyword
        CurrentValue    = $Setting.RegistryValue
        DefaultValue    = $Setting.DefaultRegistryValue
        Description     = $Setting.DisplayName
        DisplayName     = $Setting.DisplayName
        DisplayValue    = $Setting.DisplayValue
        Minihelp        = $SettingMiniHelp
        Name            = $AdapterName
        ParentId        = $Setting.InstanceID.split(':')[0]
        RegistryKeyword = $Setting.RegistryKeyword
        RegistryValue   = $Setting.RegistryValue
        Min             = $Setting.NumericParameterMinValue
        Max             = $Setting.NumericParameterMaxValue
        Step            = $Setting.NumericParameterStepValue
        }
}

function GetSettingMinihelp($Setting)
{
    $SettingMiniHelp = $MiniHelp.$($Setting.RegistryKeyword)
    if ([string]::IsNullOrEmpty($SettingMiniHelp))
    {
        $SettingMiniHelp = $Messages.MiniHelpNotFound
    }

    return $SettingMiniHelp
}

# SIG # Begin signature block
# MIIotwYJKoZIhvcNAQcCoIIoqDCCKKQCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCDnRqPyGB1HV7S3
# oBgFYFhlsnLZklGwceiQ+p+0wRnoxKCCEgUwggWeMIIEhqADAgECAhEAzS1l4rws
# CIvYBjRVawV4ujANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQGEwJHQjEbMBkGA1UE
# CBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYDVQQHEwdTYWxmb3JkMRgwFgYDVQQK
# Ew9TZWN0aWdvIExpbWl0ZWQxJDAiBgNVBAMTG1NlY3RpZ28gUlNBIENvZGUgU2ln
# bmluZyBDQTAeFw0yMTA0MDIwMDAwMDBaFw0yMzA0MDIyMzU5NTlaMIGEMQswCQYD
# VQQGEwJVUzETMBEGA1UECAwKQ2FsaWZvcm5pYTEUMBIGA1UEBwwLU2FudGEgQ2xh
# cmExGjAYBgNVBAoMEUludGVsIENvcnBvcmF0aW9uMRIwEAYDVQQLDAlTb2xhcktN
# Q1MxGjAYBgNVBAMMEUludGVsIENvcnBvcmF0aW9uMIIBojANBgkqhkiG9w0BAQEF
# AAOCAY8AMIIBigKCAYEA7CCN9iKpDHOrRceKhlXFP/tf6Lllw2H2fR9KVI4/fQIx
# MU1hXwnlHmAzMCY7IgcCFY4p3F5/MJGKaqYngwOo28Zo6Q1N6ukysA7PSavmF2RY
# WD6VFeya/2H0PoNeRFjHaRzSeynFFeJAFew9r7UReUwM/507sxZYPQuWWIdAEK7H
# Dqp2VlHmgZOXVGHhNO6GFOKpC/C01g6X3x6OquddRNMt5UrZzZzDo5MpJz9SBB2V
# jiqwZ80dvNR2W2xi90cIHh4BkXvB54UNkp4VTVu16T0k3cweo+C39U7GrCAr5Axz
# DETjBvhNtP1sf9SoRV7xY6g5wssfI7yYT9J0gsifn/Vy8MWH355TPoA+PVhbAu0m
# 9FMz4EWu55nnUurNML2jaUxsos21/7ELat12kWC0tq9fhkODjKO8X9PuiBHflZLk
# d3F4QcSMvuGocWGqE77VV3vn8jlvigm2TOV0CfGTQajGMX0jeTRZ19fzBNkt2X9d
# SSGolI/Kj1gSvCggpkUBAgMBAAGjggGQMIIBjDAfBgNVHSMEGDAWgBQO4TqoUzox
# 1Yq+wbutZxoDha00DjAdBgNVHQ4EFgQUshkNuM2SdwJnW4vFy8c4FtUTrbQwDgYD
# VR0PAQH/BAQDAgeAMAwGA1UdEwEB/wQCMAAwEwYDVR0lBAwwCgYIKwYBBQUHAwMw
# EQYJYIZIAYb4QgEBBAQDAgQQMEoGA1UdIARDMEEwNQYMKwYBBAGyMQECAQMCMCUw
# IwYIKwYBBQUHAgEWF2h0dHBzOi8vc2VjdGlnby5jb20vQ1BTMAgGBmeBDAEEATBD
# BgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29S
# U0FDb2RlU2lnbmluZ0NBLmNybDBzBggrBgEFBQcBAQRnMGUwPgYIKwYBBQUHMAKG
# Mmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1JTQUNvZGVTaWduaW5nQ0Eu
# Y3J0MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG
# 9w0BAQsFAAOCAQEAVadLNRW4f/pKMqrbn0BdOoQ8/1EJ87gvVfosei2bLwTEvpmv
# mn2n561H6AFedtIJ6L4FmXII4M4r20i+5LREbI6PpKDmOAf4xW7POxfCRvkTQAZO
# 3zoVxjMQBXo7cZVF1xHCdviXzD1usuIiCF8DLm6z4O/kyeFFNcn816yPQct91Pnk
# SBBVvL+Kwu8xvR+ZIQy632WUA4HnNpRdFnVSzUifEg2GrtsKZR8k+rm2o8K8yjJq
# 3SznwgJQCMVMh3CtRtUwE/c7o/6rvm53fTYJDd3aoPHVgH6S2WqS3+3mQG7A6hTD
# nrP/mYnS4PF7XzxxjZhUlhy4G/MarJPvT9IrNDCCBfUwggPdoAMCAQICEB2iSDBv
# myYY0ILgln0z02owDQYJKoZIhvcNAQEMBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0eTEeMBwGA1UEChMV
# VGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1c3QgUlNBIENl
# cnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTE4MTEwMjAwMDAwMFoXDTMwMTIzMTIz
# NTk1OVowfDELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFuY2hlc3Rl
# cjEQMA4GA1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSQw
# IgYDVQQDExtTZWN0aWdvIFJTQSBDb2RlIFNpZ25pbmcgQ0EwggEiMA0GCSqGSIb3
# DQEBAQUAA4IBDwAwggEKAoIBAQCGIo0yhXoYn0nwli9jCB4t3HyfFM/jJrYlZilA
# hlRGdDFixRDtsocnppnLlTDAVvWkdcapDlBipVGREGrgS2Ku/fD4GKyn/+4uMyD6
# DBmJqGx7rQDDYaHcaWVtH24nlteXUYam9CflfGqLlR5bYNV+1xaSnAAvaPeX7Wpy
# vjg7Y96Pv25MQV0SIAhZ6DnNj9LWzwa0VwW2TqE+V2sfmLzEYtYbC43HZhtKn52B
# xHJAteJf7wtF/6POF6YtVbC3sLxUap28jVZTxvC6eVBJLPcDuf4vZTXyIuosB69G
# 2flGHNyMfHEo8/6nxhTdVZFuihEN3wYklX0Pp6F8OtqGNWHTAgMBAAGjggFkMIIB
# YDAfBgNVHSMEGDAWgBRTeb9aqitKz1SA4dibwJ3ysgNmyzAdBgNVHQ4EFgQUDuE6
# qFM6MdWKvsG7rWcaA4WtNA4wDgYDVR0PAQH/BAQDAgGGMBIGA1UdEwEB/wQIMAYB
# Af8CAQAwHQYDVR0lBBYwFAYIKwYBBQUHAwMGCCsGAQUFBwMIMBEGA1UdIAQKMAgw
# BgYEVR0gADBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8vY3JsLnVzZXJ0cnVzdC5j
# b20vVVNFUlRydXN0UlNBQ2VydGlmaWNhdGlvbkF1dGhvcml0eS5jcmwwdgYIKwYB
# BQUHAQEEajBoMD8GCCsGAQUFBzAChjNodHRwOi8vY3J0LnVzZXJ0cnVzdC5jb20v
# VVNFUlRydXN0UlNBQWRkVHJ1c3RDQS5jcnQwJQYIKwYBBQUHMAGGGWh0dHA6Ly9v
# Y3NwLnVzZXJ0cnVzdC5jb20wDQYJKoZIhvcNAQEMBQADggIBAE1jUO1HNEphpNve
# aiqMm/EAAB4dYns61zLC9rPgY7P7YQCImhttEAcET7646ol4IusPRuzzRl5ARokS
# 9At3WpwqQTr81vTr5/cVlTPDoYMot94v5JT3hTODLUpASL+awk9KsY8k9LOBN9O3
# ZLCmI2pZaFJCX/8E6+F0ZXkI9amT3mtxQJmWunjxucjiwwgWsatjWsgVgG10Xkp1
# fqW4w2y1z99KeYdcx0BNYzX2MNPPtQoOCwR/oEuuu6Ol0IQAkz5TXTSlADVpbL6f
# ICUQDRn7UJBhvjmPeo5N9p8OHv4HURJmgyYZSJXOSsnBf/M6BZv5b9+If8AjntIe
# Q3pFMcGcTanwWbJZGehqjSkEAnd8S0vNcL46slVaeD68u28DECV3FTSK+TbMQ5Lk
# uk/xYpMoJVcp+1EZx6ElQGqEV8aynbG8HArafGd+fS7pKEwYfsR7MUFxmksp7As9
# V1DSyt39ngVR5UR43QHesXWYDVQk/fBO4+L4g71yuss9Ou7wXheSaG3IYfmm8SoK
# C6W59J7umDIFhZ7r+YMp08Ysfb06dy6LN0KgaoLtO0qqlBCk4Q34F8W2WnkzGJLj
# tXX4oemOCiUe5B7xn1qHI/+fpFGe+zmAEc3btcSnqIBv5VPU4OOiwtJbGvoyJi1q
# V3AcPKRYLqPzW0sH3DJZ84enGm1YMIIGZjCCBE6gAwIBAgITMwAAAES3P/zvWs+i
# egAAAAAARDANBgkqhkiG9w0BAQUFADB/MQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMSkwJwYDVQQDEyBNaWNyb3NvZnQgQ29kZSBWZXJpZmljYXRp
# b24gUm9vdDAeFw0xNTA3MjIyMTAzNDlaFw0yNTA3MjIyMTAzNDlaMIGIMQswCQYD
# VQQGEwJVUzETMBEGA1UECBMKTmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENp
# dHkxHjAcBgNVBAoTFVRoZSBVU0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNF
# UlRydXN0IFJTQSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eTCCAiIwDQYJKoZIhvcN
# AQEBBQADggIPADCCAgoCggIBAIASZRc2DsPbCLPQrFcNdu3NJ9NMrVCDYeKqIE0J
# LWQJ3M6Jn8w9qez2z8Hc8dOx1ns3KBErR9o5xrw6GbRfpr19naNjQrZ28qk7K5H4
# 4m/Q7BYgkAk+4uh0yRi0kdRiZNt/owbxiBhqkCI8vP4T8IcUe/bkH47U5FHGEWdG
# CFHLhhRUP7wz/n5snP8WnRi9UY41pqdmyHJn2yFmsdSbeAPAUDrozPDcvJ5M/q8F
# ljUfV1q3/875PbcstvZU3cjnEjpNrkyKt1yatLcgPcp/IjSufjtoZgFE5wFORlOb
# M2D3lL5TN5BzQ/Myw1Pv26r+dE5px2uMYJPexMcM3+EyrsyTO1F4lWeL7j1W/gzQ
# aQ8bD/MlJmszbfduR/pzQ+V+DqVmsSl8MoRjVYnEDcGTVDAZE6zTfTen6106bDVc
# 20HXEtqpSQvf2ICKCZNijrVmzyWIzYS4sT+kOQ/ZAp7rEkyVfPNrBaleFoPMuGfi
# 6BOdzFuC00yz7Vv/3uVzrCM7LQC/NVV0CUnYSVgaf5I25lGSDvMmfRxNF7zJ7EMm
# 0L9BX0CpRET0medXh55QH1dUqD79dGMvsVBlCeZYQi5DGky08CVHWfoEHpPUJkZK
# UIGy3r54t/xnFeHJV4QeD2PW6WK61l9VLupcxigIBCU5uA4rqfJMlxwHPw1S9e3v
# L4IPAgMBAAGjgdAwgc0wEwYDVR0lBAwwCgYIKwYBBQUHAwMwEgYDVR0TAQH/BAgw
# BgEB/wIBAjAdBgNVHQ4EFgQUU3m/WqorSs9UgOHYm8Cd8rIDZsswCwYDVR0PBAQD
# AgGGMB8GA1UdIwQYMBaAFGL7CiFbf0NuEdoJVFBr9dKWcfGeMFUGA1UdHwROMEww
# SqBIoEaGRGh0dHA6Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3Rz
# L01pY3Jvc29mdENvZGVWZXJpZlJvb3QuY3JsMA0GCSqGSIb3DQEBBQUAA4ICAQBr
# IpM8PTlUcWRrDvLkPDARxSBKS4YPkvH/M3k62eSYpw5AoCKAfmGy4KcZzyaVMSpl
# 1GpPMYbqwMYuxWSMPUhZzQsvdD2UJhMQQtSXmCdePHbSeGkdGmTnBXJ14OtmQEOf
# jwxG/5dgpshnrRAIm2Km6b46itMHTZ9ykyW8BhHgLJA4Pmcc/RnXnpDOPcLg52Gs
# wOUE9R6ZVAyRDQFWcTeuJ9SeQyKlySfNTeVxEjkkpUFWh/+8VRQPJcqJ7seX5dIT
# /z1+GqCPP8gs16Nw0MdgwPzYPlHnl8Y+O+3PeL6KyuPE8qen7Z6uCAKPoFLbch7V
# O8NNn476m3DH+OO/bD+Sm+Q3PuxqjCn5waK/iz4aaWb7HGNPJgHJAsQ+0v/DQ6gb
# /Zn61LylueKTLzsBxdH0Oi9ow+Bkt1qVXkbMB4NpuzwFklZzNXNFmE582BKlt0Lp
# omP2QmAYcNE7bzHAh8fmceHzRhbp9bhys+ltH2ImSaNJi91ox4toVvfe/PqHJLgD
# gReP5fFnah2u03T3jKVdswuOQimWzknEd35mfAEXGmwUJMOwF3cF2BpAt4Zr2OR7
# QKx+305vJPkggIKMM+fl+inYndqLcF0ryR2CTAtny4RBnucGfhGDRC2KGe70f5rd
# eRw3GR6fP4wpug1cEIY3bEjNRV3NcLy80U1d2MW4djGCFggwghYEAgEBMIGRMHwx
# CzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNV
# BAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEkMCIGA1UEAxMb
# U2VjdGlnbyBSU0EgQ29kZSBTaWduaW5nIENBAhEAzS1l4rwsCIvYBjRVawV4ujAN
# BglghkgBZQMEAgEFAKBqMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisG
# AQQBgjcCAQsxDjAMBgorBgEEAYI3AgEWMC8GCSqGSIb3DQEJBDEiBCCcB2ZcPyST
# DBwDLnnE5yIvQIDcGcvWv2vOSYtHo2FKiTANBgkqhkiG9w0BAQEFAASCAYBqgj+H
# TedS5sLQ+gI/ZkslXjM5MXfk7Is8ZDMrPwxrx1+CsVAQjy5m4hGhnbG0UVt6FC8f
# 8Cr0yVsXQiMQhNa/N06qrszHwa3ZP0qcyEeajA7FOuZcvnpwuwtj9/wTiZOK2jhu
# YLLAK7Q4VfTI29tJoLAIKaepXsxp4xOOjkhWNZmMJqkHtJBOfVmtwqtM0Dx2GWiN
# 1PCzrTEQvkBOm1sv76JDpUl34m3bKavYILrZSk2d0Vj3dKfSJgMdK+8wzGCQZ9GQ
# Ve4TP4MO0CfdpXDrut97dUxmF3OKtdlqe+ElVET3O6jGkS8pNP0RPKX4tMfi2RR7
# ilZM3cqmvsmxpXDQ1yv163568mreHGXkdOMRnoNqB1tcv0S+QePGjFwdBYvXY0kz
# aIUzfmI9WhHGGH5QknKTiiW7mU5foSiNr8g89E5JbATy8vh2NI+hHLyYa/ATNgCF
# XprjYNJuWBx2GMnOdXLuiBC9FRRZ7roZ+3GytL0rG7xfyftmw4aCBhQy8NShghNb
# MIITVwYKKwYBBAGCNwMDATGCE0cwghNDBgkqhkiG9w0BBwKgghM0MIITMAIBAzEP
# MA0GCWCGSAFlAwQCAgUAMIH6BgsqhkiG9w0BCRABBKCB6gSB5zCB5AIBAQYKKwYB
# BAGyMQIBATAxMA0GCWCGSAFlAwQCAQUABCBA/jmbVrU/eGUhCdpFkvNhg4QuQ2SN
# eGkkT2TYjQttnQIVAOVD2NwooAjsoAueSWJyHzAq5QaCGA8yMDIyMDYwMTIyNDQw
# OVoCCFcP/sTZ+0E6oG6kbDBqMQswCQYDVQQGEwJHQjETMBEGA1UECBMKTWFuY2hl
# c3RlcjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSwwKgYDVQQDDCNTZWN0aWdv
# IFJTQSBUaW1lIFN0YW1waW5nIFNpZ25lciAjM6CCDeowggb2MIIE3qADAgECAhEA
# kDl/mtJKOhPyvZFfCDipQzANBgkqhkiG9w0BAQwFADB9MQswCQYDVQQGEwJHQjEb
# MBkGA1UECBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYDVQQHEwdTYWxmb3JkMRgw
# FgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxJTAjBgNVBAMTHFNlY3RpZ28gUlNBIFRp
# bWUgU3RhbXBpbmcgQ0EwHhcNMjIwNTExMDAwMDAwWhcNMzMwODEwMjM1OTU5WjBq
# MQswCQYDVQQGEwJHQjETMBEGA1UECBMKTWFuY2hlc3RlcjEYMBYGA1UEChMPU2Vj
# dGlnbyBMaW1pdGVkMSwwKgYDVQQDDCNTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5n
# IFNpZ25lciAjMzCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAJCycT95
# 4dS5ihfMw5fCkJRy7Vo6bwFDf3NaKJ8kfKA1QAb6lK8KoYO2E+RLFQZeaoogNHF7
# uyWtP1sKpB8vbH0uYVHQjFk3PqZd8R5dgLbYH2DjzRJqiB/G/hjLk0NWesfOA9YA
# ZChWIrFLGdLwlslEHzldnLCW7VpJjX5y5ENrf8mgP2xKrdUAT70KuIPFvZgsB3YB
# cEXew/BCaer/JswDRB8WKOFqdLacRfq2Os6U0R+9jGWq/fzDPOgNnDhm1fx9HptZ
# jJFaQldVUBYNS3Ry7qAqMfwmAjT5ZBtZ/eM61Oi4QSl0AT8N4BN3KxE8+z3N0Ofh
# l1tV9yoDbdXNYtrOnB786nB95n1LaM5aKWHToFwls6UnaKNY/fUta8pfZMdrKAza
# rHhB3pLvD8Xsq98tbxpUUWwzs41ZYOff6Bcio3lBYs/8e/OS2q7gPE8PWsxu3x+8
# Iq+3OBCaNKcL//4dXqTz7hY4Kz+sdpRBnWQd+oD9AOH++DrUw167aU1ymeXxMi1R
# +mGtTeomjm38qUiYPvJGDWmxt270BdtBBcYYwFDk+K3+rGNhR5G8RrVGU2zF9OGG
# J5OEOWx14B0MelmLLsv0ZCxCR/RUWIU35cdpp9Ili5a/xq3gvbE39x/fQnuq6xzp
# 6z1a3fjSkNVJmjodgxpXfxwBws4cfcz7lhXFAgMBAAGjggGCMIIBfjAfBgNVHSME
# GDAWgBQaofhhGSAPw0F3RSiO0TVfBhIEVTAdBgNVHQ4EFgQUJS5oPGuaKyQUqR+i
# 3yY6zxSm8eAwDgYDVR0PAQH/BAQDAgbAMAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/
# BAwwCgYIKwYBBQUHAwgwSgYDVR0gBEMwQTA1BgwrBgEEAbIxAQIBAwgwJTAjBggr
# BgEFBQcCARYXaHR0cHM6Ly9zZWN0aWdvLmNvbS9DUFMwCAYGZ4EMAQQCMEQGA1Ud
# HwQ9MDswOaA3oDWGM2h0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1JTQVRp
# bWVTdGFtcGluZ0NBLmNybDB0BggrBgEFBQcBAQRoMGYwPwYIKwYBBQUHMAKGM2h0
# dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1JTQVRpbWVTdGFtcGluZ0NBLmNy
# dDAjBggrBgEFBQcwAYYXaHR0cDovL29jc3Auc2VjdGlnby5jb20wDQYJKoZIhvcN
# AQEMBQADggIBAHPa7Whyy8K5QKExu7QDoy0UeyTntFsVfajp/a3Rkg18PTagadnz
# mjDarGnWdFckP34PPNn1w3klbCbojWiTzvF3iTl/qAQF2jTDFOqfCFSr/8R+lmwr
# 05TrtGzgRU0ssvc7O1q1wfvXiXVtmHJy9vcHKPPTstDrGb4VLHjvzUWgAOT4BHa7
# V8WQvndUkHSeC09NxKoTj5evATUry5sReOny+YkEPE7jghJi67REDHVBwg80uIid
# yCLxE2rbGC9ueK3EBbTohAiTB/l9g/5omDTkd+WxzoyUbNsDbSgFR36bLvBk+9uk
# AzEQfBr7PBmA0QtwuVVfR745ZM632iNUMuNGsjLY0imGyRVdgJWvAvu00S6dOHw1
# 4A8c7RtHSJwialWC2fK6CGUD5fEp80iKCQFMpnnyorYamZTrlyjhvn0boXztVoCm
# 9CIzkOSEU/wq+sCnl6jqtY16zuTgS6Ezqwt2oNVpFreOZr9f+h/EqH+noUgUkQ2C
# /L1Nme3J5mw2/ndDmbhpLXxhL+2jsEn+W75pJJH/k/xXaZJL2QU/bYZy06LQwGTS
# OkLBGgP70O2aIbg/r6ayUVTVTMXKHxKNV8Y57Vz/7J8mdq1kZmfoqjDg0q23fbFq
# QSduA4qjdOCKCYJuv+P2t7yeCykYaIGhnD9uFllLFAkJmuauv2AV3Yb1MIIG7DCC
# BNSgAwIBAgIQMA9vrN1mmHR8qUY2p3gtuTANBgkqhkiG9w0BAQwFADCBiDELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCk5ldyBKZXJzZXkxFDASBgNVBAcTC0plcnNleSBD
# aXR5MR4wHAYDVQQKExVUaGUgVVNFUlRSVVNUIE5ldHdvcmsxLjAsBgNVBAMTJVVT
# RVJUcnVzdCBSU0EgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkwHhcNMTkwNTAyMDAw
# MDAwWhcNMzgwMTE4MjM1OTU5WjB9MQswCQYDVQQGEwJHQjEbMBkGA1UECBMSR3Jl
# YXRlciBNYW5jaGVzdGVyMRAwDgYDVQQHEwdTYWxmb3JkMRgwFgYDVQQKEw9TZWN0
# aWdvIExpbWl0ZWQxJTAjBgNVBAMTHFNlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcg
# Q0EwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDIGwGv2Sx+iJl9AZg/
# IJC9nIAhVJO5z6A+U++zWsB21hoEpc5Hg7XrxMxJNMvzRWW5+adkFiYJ+9UyUnku
# yWPCE5u2hj8BBZJmbyGr1XEQeYf0RirNxFrJ29ddSU1yVg/cyeNTmDoqHvzOWEnT
# v/M5u7mkI0Ks0BXDf56iXNc48RaycNOjxN+zxXKsLgp3/A2UUrf8H5VzJD0BKLwP
# DU+zkQGObp0ndVXRFzs0IXuXAZSvf4DP0REKV4TJf1bgvUacgr6Unb+0ILBgfrhN
# 9Q0/29DqhYyKVnHRLZRMyIw80xSinL0m/9NTIMdgaZtYClT0Bef9Maz5yIUXx7gp
# GaQpL0bj3duRX58/Nj4OMGcrRrc1r5a+2kxgzKi7nw0U1BjEMJh0giHPYla1IXMS
# Hv2qyghYh3ekFesZVf/QOVQtJu5FGjpvzdeE8NfwKMVPZIMC1Pvi3vG8Aij0bdon
# igbSlofe6GsO8Ft96XZpkyAcSpcsdxkrk5WYnJee647BeFbGRCXfBhKaBi2fA179
# g6JTZ8qx+o2hZMmIklnLqEbAyfKm/31X2xJ2+opBJNQb/HKlFKLUrUMcpEmLQTkU
# Ax4p+hulIq6lw02C0I3aa7fb9xhAV3PwcaP7Sn1FNsH3jYL6uckNU4B9+rY5WDLv
# bxhQiddPnTO9GrWdod6VQXqngwIDAQABo4IBWjCCAVYwHwYDVR0jBBgwFoAUU3m/
# WqorSs9UgOHYm8Cd8rIDZsswHQYDVR0OBBYEFBqh+GEZIA/DQXdFKI7RNV8GEgRV
# MA4GA1UdDwEB/wQEAwIBhjASBgNVHRMBAf8ECDAGAQH/AgEAMBMGA1UdJQQMMAoG
# CCsGAQUFBwMIMBEGA1UdIAQKMAgwBgYEVR0gADBQBgNVHR8ESTBHMEWgQ6BBhj9o
# dHRwOi8vY3JsLnVzZXJ0cnVzdC5jb20vVVNFUlRydXN0UlNBQ2VydGlmaWNhdGlv
# bkF1dGhvcml0eS5jcmwwdgYIKwYBBQUHAQEEajBoMD8GCCsGAQUFBzAChjNodHRw
# Oi8vY3J0LnVzZXJ0cnVzdC5jb20vVVNFUlRydXN0UlNBQWRkVHJ1c3RDQS5jcnQw
# JQYIKwYBBQUHMAGGGWh0dHA6Ly9vY3NwLnVzZXJ0cnVzdC5jb20wDQYJKoZIhvcN
# AQEMBQADggIBAG1UgaUzXRbhtVOBkXXfA3oyCy0lhBGysNsqfSoF9bw7J/RaoLlJ
# WZApbGHLtVDb4n35nwDvQMOt0+LkVvlYQc/xQuUQff+wdB+PxlwJ+TNe6qAcJlhc
# 87QRD9XVw+K81Vh4v0h24URnbY+wQxAPjeT5OGK/EwHFhaNMxcyyUzCVpNb0llYI
# uM1cfwGWvnJSajtCN3wWeDmTk5SbsdyybUFtZ83Jb5A9f0VywRsj1sJVhGbks8Vm
# Bvbz1kteraMrQoohkv6ob1olcGKBc2NeoLvY3NdK0z2vgwY4Eh0khy3k/ALWPncE
# vAQ2ted3y5wujSMYuaPCRx3wXdahc1cFaJqnyTdlHb7qvNhCg0MFpYumCf/RoZSm
# Tqo9CfUFbLfSZFrYKiLCS53xOV5M3kg9mzSWmglfjv33sVKRzj+J9hyhtal1H3G/
# W0NdZT1QgW6r8NDT/LKzH7aZlib0PHmLXGTMze4nmuWgwAxyh8FuTVrTHurwROYy
# bxzrF06Uw3hlIDsPQaof6aFBnf6xuKBlKjTg3qj5PObBMLvAoGMs/FwWAKjQxH/q
# EZ0eBsambTJdtDgJK0kHqv3sMNrxpy/Pt/360KOE2See+wFmd7lWEOEgbsausfm2
# usg1XTN2jvF8IAwqd661ogKGuinutFoAsYyr4/kKyVRd1LlqdJ69SK6YMYIELTCC
# BCkCAQEwgZIwfTELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFuY2hl
# c3RlcjEQMA4GA1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVk
# MSUwIwYDVQQDExxTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5nIENBAhEAkDl/mtJK
# OhPyvZFfCDipQzANBglghkgBZQMEAgIFAKCCAWswGgYJKoZIhvcNAQkDMQ0GCyqG
# SIb3DQEJEAEEMBwGCSqGSIb3DQEJBTEPFw0yMjA2MDEyMjQ0MDlaMD8GCSqGSIb3
# DQEJBDEyBDAONFvKPHDJF+hsRjsLQLjw72oyzPhMKwL7TudI9eA9XS9ebJKImm6V
# uGBq5SvLYT0wge0GCyqGSIb3DQEJEAIMMYHdMIHaMIHXMBYEFKs0ATqsQJcxnwga
# 8LMY4YP4D3iBMIG8BBQC1luV4oNwwVcAlfqI+SPdk3+tjzCBozCBjqSBizCBiDEL
# MAkGA1UEBhMCVVMxEzARBgNVBAgTCk5ldyBKZXJzZXkxFDASBgNVBAcTC0plcnNl
# eSBDaXR5MR4wHAYDVQQKExVUaGUgVVNFUlRSVVNUIE5ldHdvcmsxLjAsBgNVBAMT
# JVVTRVJUcnVzdCBSU0EgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkCEDAPb6zdZph0
# fKlGNqd4LbkwDQYJKoZIhvcNAQEBBQAEggIADh/KbBIYTJHlYan2BcW1JsmbrHpk
# jsRTKXsYkRHrNQJ/wFGrt3Kp64k9MuPeTes5AW08MObOdpr8rxJ6kZP3zQHUmQ++
# lKOckja1DLsO1/H8m42zRuhki6Rn7nkvCqwA2BB0fDtNhwYkYQyyHWpp4KMZAlEr
# 8uu0rh9Lvznww+MoVX1koJKSxu3iTfdedriIsrrhk54lnByvzBsasTxd3yyE1dZQ
# nboRBrXwa4Zq+c4wj9UOAZiCnqsHP+tyeeBmd+SxBjTEzCe30ImnDoH8SBlgtBkM
# /5HppihE8Gd2bEbF8+OHiB25YPadqvZ000jJAQrXqagNr/Dg+K8s7Uzds+0ImOwI
# 6gjzccpHRnRKy+YVY/ltTPZvajVN2HzyPnYBZaW6Njy2rfZ56R3TgARXIajH/AdT
# PhX20onppZ2JRcZ93u13Nz9TdiO5Xv5ObXwoSCUOIN3bnhBCpjGK2P7CnKqhvlW+
# lAaJmvVbgKh/xdzOknXP+xLZ/l8LCUZ/4nDeFQw6kG8oBNYAe35IXOg/arxa72PQ
# SUa/gEs78rgu3D4kQmo1Qh+44VJ3k55ey1tv4a9Fmm8G3pMt0c2NbWit9ssCwVSg
# itXSHF/9S+BDAQ9niWwCnq2+8hXlXx3V3FAHhDoI04UUJ1h1VCT4BLNRFv3bG9Bo
# gNmlyvz5HXbWCuw=
# SIG # End signature block
