/******************************************************************************

  Copyright (c) 2013-2019, Intel Corporation
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.

   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.

   3. Neither the name of the Intel Corporation nor the names of its
      contributors may be used to endorse or promote products derived from
      this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  POSSIBILITY OF SUCH DAMAGE.

******************************************************************************/
/*$FreeBSD$*/
/**
 * @file iavf_compat_iflib.h
 * @brief kernel compatibility macros for the iflib interface
 *
 * Contains macro definitions to aid in compatibility between various FreeBSD
 * kernel versions, specific to the iflib interface.
 */
#ifndef _IAVF_COMPAT_IFLIB_H_
#define _IAVF_COMPAT_IFLIB_H_

#include "iavf_compat_common.h"

#include <net/ethernet.h>
#include <net/iflib.h>
#include "ifdi_if.h"

/*
 * Over the course of its development, iflib has made a few API changes, which
 * we need to work around here in order to allow the driver code to work on
 * different versions of the FreeBSD kernel.
 */

#if __FreeBSD_version < 1102000
/**
 * @typedef qidx_t
 * @brief queue index type
 *
 * New versions of iflib introduce the qidx_t type for representing an index
 * into a queue. Backport it as a uint16_t (the type it replaced) so that we
 * can use qidx_t everywhere.
 */
typedef uint16_t qidx_t;
#define QIDX_INVALID 0xFFFF
#else
#define HAVE_RXD_REFILL_USES_RXD_UPDATE_T
#define HAVE_TXRX_OPS_USE_QIDX_T
#define HAVE_IFDI_TX_QUEUE_INTR_ENABLE
#endif

#if __FreeBSD_version >= 1102000
#define HAVE_TXRX_OPS_IN_SOFTC_CTX
#define HAVE_CAPENABLE_IN_SOFTC_CTX
#endif

#if __FreeBSD_version >= 1102000
#define HAVE_IFLIB_SOFTIRQ_ALLOC_USES_IRQ
#endif

#if __FreeBSD_version >= 1200073
#define HAVE_TSO_MAX_IN_SHARED_CTX
#define HAVE_CAPABILITIES_IN_SOFTC_CTX
#endif

#define MOD_DEP_iflib _ICE_MOD_DEP(iflib)
#if __FreeBSD_version < 1102000
#define IAVF_MAGIC (int)IFLIB_MAGIC
#else
#define IAVF_MAGIC IFLIB_MAGIC
#endif

#if __FreeBSD_version >= 1200085
#define HAVE_IFLIB_IN_DETACH
#endif

/*
 * Older versions of iflib do not report the TCP header length when requesting
 * that a driver perform TCP checksum offloading. Unfortunately, ice hardware
 * requires the TCP length in order to correctly compute the checksum.
 */
#if __FreeBSD_version < 1102503
#define MISSING_TCP_HLEN_FOR_CSUM_IP_TCP
#endif

/*
 * Older versions of iflib do not correctly enable work arounds necessary for
 * TSO packets to behave properly on our hardware. Check if IFLIB_TSO_INIT_IP
 * exists, and if not, disable TSOs. (Note, this flag was backported to 11.2
 * in FreeBSD_version 1101515)
 */
#ifndef IFLIB_TSO_INIT_IP
#define IFLIB_TSO_INIT_IP (0) /* since we set this in the isc_flags */
#define MISSING_WORKAROUNDS_FOR_CSUM_IP_TSO
#endif

/*
 * Older versions of iflib lack the iflib_request_reset function. It's not
 * possible to reliably backport this function, as we need to directly access
 * the internals of the iflib context structure. Without the ability to
 * request that iflib reset, the system administrator must intervene by
 * bringing the device down and up again. Communicate this to the system
 * administrator by printing a message in the kernel message log.
 */
#if (__FreeBSD_version < 1102509 || \
    (__FreeBSD_version >= 1200000 && __FreeBSD_version < 1200085))
static inline void
iflib_request_reset(if_ctx_t ctx)
{
	device_t dev = iflib_get_dev(ctx);

	device_printf(dev, "Unable to request an iflib reset. Bring the device down and up to restore functionality.\n");
}
#endif

#if (__FreeBSD_version < 1102509 || \
    (__FreeBSD_version >= 1200000 && __FreeBSD_version < 1200507))
/**
 * iflib_get_rx_mbuf_sz - Determine the Rx buffer size that iflib is using
 * @ctx: the iflib context structure
 *
 * Older versions of iflib do not expose the Rx buffer size used when
 * allocating mbufs. This function mimics the behavior that iflib, by checking
 * the isc_max_frame_size and switching between MCLBYTES and MJUMPAGESIZE.
 */
static inline uint32_t
iflib_get_rx_mbuf_sz(if_ctx_t ctx)
{
	if_softc_ctx_t sctx = iflib_get_softc_ctx(ctx);

	/*
	 * XXX don't set the max_frame_size to larger
	 * than the hardware can handle
	 */
	if (sctx->isc_max_frame_size <= MCLBYTES)
		return MCLBYTES;
	else
		return MJUMPAGESIZE;
}
#endif

/*
 * Pre-11.3 versions of iflib don't have iflib_in_detach(). That function is
 * used to determine whether iflib is in the process of unloading the driver so
 * that the driver doesn'tdo things that would lead to it accessing freed
 * memory or de-allocated structres. However, since this function checks a
 * private flag, it can't be re-implemented here, so just return 0 here.
 */
#if __FreeBSD_version < 1102509
static inline uint8_t
iflib_in_detach(if_ctx_t ctx)
{
	return (0);
}
#endif

/*
 * Older versions of iflib do not support always running the admin task. We'll
 * just define this flag to zero if it doesn't exist so that it is a no-op
 * when bit-wise OR'ing it into .isc_flags.
 */
#ifndef IFLIB_ADMIN_ALWAYS_RUN
#define IFLIB_ADMIN_ALWAYS_RUN (0)
#endif

#ifndef IFLIB_PNP_INFO
#define IFLIB_PNP_DESCR "U32:vendor;U32:device;U32:subvendor;U32:subdevice;U32:revision;U32:class;D:human"
#define IFLIB_PNP_INFO(b, u, t) \
    MODULE_PNP_INFO(IFLIB_PNP_DESCR, b, u, t, sizeof(t[0]), nitems(t))
#endif /* IFLIB_PNP_INFO */

/*
 * The version of iflib pre-11.2 doesn't define this flag, so it's defined here
 * if it doesn't exist. In order to work reliably, though, the driver may
 * actually need iflib to make the mbuf chain writable like it does in 11.2+.
 */
#ifndef IFLIB_NEED_SCRATCH
#define IFLIB_NEED_SCRATCH (0)
#endif

#endif /* _IAVF_COMPAT_IFLIB_H_ */
