/*************************************************************************************************************
 *                                                                                                           *
 *  Copyright (c) 2015 - 2016, Intel Corporation                                                                    *
 *                                                                                                           *
 *  Redistribution and use in source and binary forms, with or without                                       *
 *  modification, are permitted provided that the following conditions are met:                              *
 *                                                                                                           *
 *      * Redistributions of source code must retain the above copyright notice,                             *
 *        this list of conditions and the following disclaimer.                                              *
 *      * Redistributions in binary form must reproduce the above copyright                                  *
 *        notice, this list of conditions and the following disclaimer in the                                *
 *        documentation and/or other materials provided with the distribution.                               *
 *      * Neither the name of Intel Corporation nor the names of its contributors                            *
 *        may be used to endorse or promote products derived from this software                              *
 *        without specific prior written permission.                                                         *
 *                                                                                                           *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"                              *
 *  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE                                *
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE                           *
 *  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE                              *
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL                               *
 *  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR                               *
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER                               *
 *  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,                            *
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE                            *
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                                     *
 *                                                                                                           *
 *************************************************************************************************************
 *                                                                                                           *
 *  Module name:                                                                                             *
 *      freebsddriverdevice_i.c                                                                              *
 *                                                                                                           *
 *  Abstract:                                                                                                *
 *      This file contains freebsddriverdevice_i.c                                                           *
 *                                                                                                           *
 ************************************************************************************************************/
#include <naltypes.h>
#include <nalcodes.h>
#include <freebsdnaldriver.h>
#include <freebsddevice_i.h>
#include <freebsdnalioctl.h>
#include <nalioctl.h>

#include <sys/bus.h>
#include <dev/pci/pcireg.h>
#include <dev/pci/pcivar.h>

#include <sys/param.h>
#include <sys/bus.h>

#include <machine/bus.h>
#include <sys/rman.h>
#include <machine/resource.h>

#include <sys/systm.h>
#include <freebsddefs.h>
#include <freebsdtypes.h>

NAL_STATUS
_NalEnableDevice(
    IN      NAL_DEVICE_LOCATION     DeviceLocation,
    OUT     NAL_IO_RESOURCE*        IoResource,
    OUT     KVOID**                 PDev
    );

NAL_STATUS
_NalFillDeviceResource(
    IN      NAL_DEVICE_LOCATION     DeviceLocation,
    OUT     NAL_IO_RESOURCE*        IoResource,
    OUT     KVOID**                 PDev
    );

NAL_STATUS
_NalReleaseRegions(
    IN      KVOID*   PDev
    );

extern NAL_OS_SPEC_ADAPTER_IN_USE_TABLE    Global_AdapterInUse[NAL_OS_SPEC_MAX_PCI_DEVICES];

/***************************************************************************
**
** Name:            _NalMarkAdapterInUse()
**
** Description:     This attempts to mark the adapter in use if the init flags are not
**                  shared. It returns if the driver was already marked in use and if
**                  the driver can be used.
**
** Arguments:       NalDevice   = NAL_DEVICE_LOCATION with the PCI or BUS device info.
**                  Lock        = TRUE to lock the device, FALSE to unlock it.
**
** Returns:         TRUE  = Device is in use
**                  FALSE = Device is not yet in use.
**
****************************************************************************/
BOOLEAN
_NalMarkAdapterInUse(
    IN  NAL_DEVICE_LOCATION   NalDevice,
    IN  BOOLEAN               Lock
    )
{
    UINTN   i         = 0;
    BOOLEAN CanBeUsed = FALSE;

    /* First scan all adapters in memory to see if any match the NalDevice requested */
    for(i=0; i<NAL_OS_SPEC_MAX_PCI_DEVICES; i++)
    {
        /* Check to see if this adapter slot matches our NalDevice */
        if(Global_AdapterInUse[i].DeviceLocation.Reserved == NalDevice.Reserved)
        {
            /* See if the matching device is already marked "In Use" */
            if(Global_AdapterInUse[i].InUse == TRUE)
            {
                /* The device is in use, determine if we're being asked to "unlock" it */
                if(Lock == FALSE)
                {
                    Global_AdapterInUse[i].InUse = FALSE;
                    Global_AdapterInUse[i].DeviceLocation.Reserved = 0;
                    CanBeUsed = TRUE;
                    break;
                }
                else
                {
                    CanBeUsed = FALSE;
                    break;
                }
            }

            /* Slot found, we're asking to mark in use, and the dev is not in use */
            else if(Lock == TRUE)
            {
                Global_AdapterInUse[i].InUse = TRUE;
                CanBeUsed = TRUE;
                break;
            }
        }
    }

    /* Entry not found and we're looking to add one. */
    if(i == NAL_OS_SPEC_MAX_PCI_DEVICES && Lock == TRUE)
    {
        /* Look for a free slot */
        for(i=0; i<NAL_OS_SPEC_MAX_PCI_DEVICES; i++)
        {
            if(Global_AdapterInUse[i].DeviceLocation.Reserved == 0)
            {
                break;
            }
        }

        /* This only runs if a free slot was located */
        if(i<NAL_OS_SPEC_MAX_PCI_DEVICES)
        {
            Global_AdapterInUse[i].DeviceLocation.Reserved = NalDevice.Reserved;
            Global_AdapterInUse[i].InUse = Lock;
            CanBeUsed = TRUE;
        }
    }

    /* This happens if the adapter is requested to be unlocked but doesnt exist */
    else if(i == NAL_OS_SPEC_MAX_PCI_DEVICES && Lock == FALSE)
    {
        CanBeUsed = TRUE;
    }

    return CanBeUsed;
}

/***************************************************************************
**
** Name:            _NalIsAdapterInUse()
**
** Description:     This searches to see if the adapter is already locked as in use.
**
** Arguments:       NalDevice   = NAL_DEVICE_LOCATION with the PCI or BUS device info.
**
** Returns:         TRUE  = device can be used
**                  FALSE = device cannot be used.
**
****************************************************************************/
BOOLEAN
_NalIsAdapterInUse(
    IN  NAL_DEVICE_LOCATION   NalDevice
    )
{
    UINTN   i         = 0;
    BOOLEAN IsInUse   = FALSE;

    /* First scan all adapters in memory to see if any match the NalDevice requested */
    for(i=0; i<NAL_OS_SPEC_MAX_PCI_DEVICES; i++)
    {
        /* Check to see if this adapter slot matches our NalDevice */
        if(Global_AdapterInUse[i].DeviceLocation.Reserved == NalDevice.Reserved)
        {
            /* See if the matching device is already marked "In Use" */
            if(Global_AdapterInUse[i].InUse == TRUE)
            {
                IsInUse = TRUE;
                break;
            }
        }
    }

    return IsInUse;
}
/***************************************************************************
**
** Name:            _NalInitializeInterrupts()
**
** Description:     Tells NAL to be ready to perform the interrupt test.
**
** Arguments:       NalIsrDevice - Pointer to structure needed for the ISR Registration.
**
** Returns:         NAL_SUCCESS = ServiceRoutine registered at Vector successfully.
**                  NAL_VECTOR_INITIALIZATION_FAILED = ISR was unable to be registered.
**
****************************************************************************/
NAL_STATUS
_NalInitializeInterrupts(
    IN      NAL_OS_SPEC_ISR_DEVICE*  NalIsrDevice
    )
{
    return NAL_NOT_IMPLEMENTED;
}

/***************************************************************************
**
** Name:            _NalUninitializeInterrupts()
**
** Description:     Informs the NAL that the interrupt test is over and any
**                  ISR registration can be released.
**
** Arguments:       NalIsrDevice - Pointer to structure needed for the ISR deregistration.
**
** Returns:         NAL_SUCCESS = Vector has been uninitialized successfully.
**                  NAL_INVALID_VECTOR = Vector was never initialized.
**
****************************************************************************/
NAL_STATUS
_NalUninitializeInterrupts(
    IN      NAL_OS_SPEC_ISR_DEVICE*  NalIsrDevice
    )
{
    return NAL_NOT_IMPLEMENTED;
}

/***************************************************************************
**
** Name:            _NalHasInterruptOccurred()
**
** Description:     Checks if an interrupt has occurred for this adapter.
**
** Arguments:       NalAdapter - Pointer to the NAL_ADAPTER_STRUCTURE.
**
** Returns:         TRUE = Vector has received an interrupt and it has been cleared.
**                  FALSE = Vector has not received an interrupt.
**
****************************************************************************/
BOOLEAN
_NalHasInterruptOccurred(
    IN      NAL_OS_SPEC_ISR_DEVICE*  NalIsrDevice
    )
{
    BOOLEAN InterruptOccurred = FALSE;

    if(NalIsrDevice != NULL && NalIsrDevice->Signature == NAL_OS_SPEC_INTERRUPT_SIGNATURE)
    {
        InterruptOccurred = NalIsrDevice->DeviceInterrupted;
    }

    return InterruptOccurred;
}

/***************************************************************************
**
** Name:            NalResolveOsSpecificIoctl()
**
** Description:     This handles IOCTL for FreeBSD specific functions that must
**                  be performed in Kernel mode and called from within user mode.
**
** Arguments:       NalIoctl - Pointer to a NAL_IOCTL_INPUT_DATA structure containing all
**                             relevant data for the IOCTL.
**
** Returns:         NAL Status Code. This is the status code for
**                  NalResolveOsSpecificIoctl, not for the individual functions called
**                  from within.
**
****************************************************************************/
NAL_STATUS
NalResolveOsSpecificIoctl(
    IN      NAL_IOCTL_INPUT_DATA*  NalIoctl
    )
{
    NAL_STATUS NalStatus = NAL_INVALID_PARAMETER;

    if(NalIoctl != NULL)
    {
        switch(NalIoctl->FunctionId)
        {
           case NAL_FREEBSD_ADAPTER_IN_USE_FUNCID:
                {
                    NAL_FREEBSD_ADAPTER_IN_USE_FUNC* FunctionData =
                        (NAL_FREEBSD_ADAPTER_IN_USE_FUNC*)(&(NalIoctl->InputBuffer));

                    if(FunctionData != NULL)
                    {
                        FunctionData->CanBeUsed = _NalMarkAdapterInUse(FunctionData->NalDevice,
                                                                       FunctionData->Locked);
                        NalStatus = NAL_SUCCESS;
                    }
                    else
                    {
                        NalStatus = NAL_INVALID_PARAMETER;
                        NalDebugPrint("NalResolveOsSpecificIoctl: NAL_FREEBSD_ADAPTER_IN_USE_FUNCID FunctionData is NULL\n");
                    }
                }
                break;

           case NAL_FREEBSD_IS_ADAPTER_IN_USE_FUNCID:
                {
                    NAL_FREEBSD_IS_ADAPTER_IN_USE_FUNC* FunctionData =
                        (NAL_FREEBSD_IS_ADAPTER_IN_USE_FUNC*)(&(NalIoctl->InputBuffer));

                    if(FunctionData != NULL)
                    {
                        FunctionData->IsInUse = _NalIsAdapterInUse(FunctionData->NalDevice);
                        NalStatus = NAL_SUCCESS;
                    }
                    else
                    {
                        NalStatus = NAL_INVALID_PARAMETER;
                        NalDebugPrint("NalResolveOsSpecificIoctl: NAL_FREEBSD_IS_ADAPTER_IN_USE_FUNCID FunctionData is NULL\n");
                    }
                }
                break;

           case NAL_FREEBSD_REQUEST_REGIONS_FUNCID:
               {
                   NAL_FREEBSD_DEVICERESOURCE_FUNC* FunctionData =
                       (NAL_FREEBSD_DEVICERESOURCE_FUNC*)(&(NalIoctl->InputBuffer));

                   if(FunctionData != NULL)
                   {
                       FunctionData->ReturnValue = _NalEnableDevice(FunctionData->DeviceLocation,
                                                                    &(FunctionData->NalIoResource[0]),
                                                                    &FunctionData->Pdev);
                       NalStatus = NAL_SUCCESS;
                   }
                   else
                   {
                       NalStatus = NAL_INVALID_PARAMETER;
                       NalDebugPrint("NalResolveOsSpecificIoctl: NAL_FREEBSD_REQUEST_REGIONS_FUNCID FunctionData is NULL\n");
                   }
               }
               break;

           case NAL_FREEBSD_RELEASE_REGIONS_FUNCID:
               {
                   NAL_FREEBSD_DEVICERESOURCE_FUNC* FunctionData =
                       (NAL_FREEBSD_DEVICERESOURCE_FUNC*)(&(NalIoctl->InputBuffer));
                   NalDebugPrint("NalResolveOsSpecificIoctl: NAL_FREEBSD_RELEASE_REGIONS_FUNCID FunctionData is NULL\n");

                   if(FunctionData != NULL)
                   {
                       FunctionData->ReturnValue = _NalReleaseRegions(FunctionData->Pdev);
                       NalStatus = NAL_SUCCESS;
                   }
                   else
                   {
                       NalStatus = NAL_INVALID_PARAMETER;
                       NalDebugPrint("NalResolveOsSpecificIoctl: NAL_FREEBSD_RELEASE_REGIONS_FUNCID FunctionData is NULL\n");
                   }
               }
               break;

           case NAL_FREEBSD_FILL_DEVICE_RESOURCE_FUNCID:
               {
                   NAL_FREEBSD_DEVICERESOURCE_FUNC* FunctionData =
                       (NAL_FREEBSD_DEVICERESOURCE_FUNC*)(&(NalIoctl->InputBuffer));

                   NalDebugPrint("NalResolveOsSpecificIoctl: Nal Fill device resource Ioctl\n");

                   if(FunctionData != NULL)
                   {
                       FunctionData->ReturnValue = _NalFillDeviceResource(FunctionData->DeviceLocation,
                                                                          &(FunctionData->NalIoResource[0]),
                                                                          &FunctionData->Pdev);
                       NalStatus = NAL_SUCCESS;
                   }
                   else
                   {
                       NalStatus = NAL_INVALID_PARAMETER;
                       NalDebugPrint("NalResolveOsSpecificIoctl: NAL_FREEBSD_FILL_DEVICE_RESOURCE_FUNCID FunctionData is NULL\n");
                   }
               }
               break;
           case NAL_FREEBSD_FREEMEMORYNONPAGEDPCI_FUNCID:
               {
                   NAL_FREEBSD_FREEMEMORYNONPAGEDPCI_FUNC* FunctionData =
                       (NAL_FREEBSD_FREEMEMORYNONPAGEDPCI_FUNC*)(&(NalIoctl->InputBuffer));

                       _NalFreeMemoryNonPagedPci(FunctionData->PDev,
                                                FunctionData->Address);

                   NalStatus = NAL_SUCCESS;
               }
               break;

           case NAL_FREEBSD_ALLOCATEMEMORYNONPAGEDPCI_FUNCID:
               {
                   NAL_FREEBSD_ALLOCATEMEMORYNONPAGEDPCI_FUNC* FunctionData =
                       (NAL_FREEBSD_ALLOCATEMEMORYNONPAGEDPCI_FUNC*)(&(NalIoctl->InputBuffer));

                       FunctionData->ReturnValue =
                           _NalAllocateMemoryNonPagedPci(FunctionData->PDev,
                                                      FunctionData->ByteCount,
                                                      FunctionData->Alignment,
                                                      &FunctionData->PhysicalAddress,
                                                      NULL,
                                                      0);

                   NalStatus = NAL_SUCCESS;
               }
               break;

           case NAL_FREEBSD_GET_DRIVER_REFCOUNT_FUNCID:
               {
                   NAL_FREEBSD_REFCOUNT_FUNCS* FunctionData =
                           (NAL_FREEBSD_REFCOUNT_FUNCS*)(&(NalIoctl->InputBuffer));

                   if(FunctionData != NULL)
                   {
                       FunctionData->ReturnValue =  _NalDriverGetReferenceCount();
                       NalStatus = NAL_SUCCESS;
                   }
                   else
                   {
                       NalStatus = NAL_INVALID_PARAMETER;
                   }
               }
               break;

           case NAL_FREEBSD_INC_DRIVER_REFCOUNT_FUNCID:
               {
                   _NalDriverIncrementReferenceCount();
                   NalStatus = NAL_SUCCESS;
               }
               break;

           case NAL_FREEBSD_DEC_DRIVER_REFCOUNT_FUNCID:
               {
                   _NalDriverDecrementReferenceCount();
                   NalStatus = NAL_SUCCESS;
               }
               break;

           case NAL_FREEBSD_DRIVER_GET_VERSION:
               {
                   NAL_FREEBSD_DRIVER_GET_VERSION_FUNCS* FunctionData =
                           (NAL_FREEBSD_DRIVER_GET_VERSION_FUNCS*)(&(NalIoctl->InputBuffer));

                   NalDebugPrint("NalResolveOsSpecificIoctl: Nal get version Ioctl\n");

                   if(FunctionData != NULL)
                   {
                       _NalDriverGetVersion(FunctionData->Version);
                       NalStatus = NAL_SUCCESS;
                   }
                   else
                   {
                       NalStatus = NAL_INVALID_PARAMETER;
                   }
               }
               break;

           default:
                NalStatus = NAL_IOCTL_INVALID_FUNCTION_ID;
                break;
        }
    }

    return NalStatus;
}

/***************************************************************************
**
** Name:            _NalEnableDevice()
**
** Description:     This does everything that is required to activate the device.
*                   This requests the I/O and Memory regions for the device.
**                  This performs what is required for I/O to be performed on FreeBSD.
**
** Arguments:       DeviceLocation - Pci location information for this device.
**
** Returns:         IoResource - List of resources.
**                  PDev - device handle in kernel.
**
**                  NAL Status Code. This is the status code for
**                  NalResolveOsSpecificIoctl, not for the individual functions called
**                  from within.
**
****************************************************************************/
NAL_STATUS
_NalEnableDevice(
    IN      NAL_DEVICE_LOCATION     DeviceLocation,
    OUT     NAL_IO_RESOURCE*        IoResource,
    OUT     KVOID**                 PDev
    )
{
    NAL_STATUS NalStatus        = NAL_INVALID_PARAMETER;
    device_t   Device           = NULL;
    UINT32     Register         = 0;
    int        Result           = -1;
    int        BarIndex         = 0;
    int        BarType          = 0;
    int        RegisterOffset   = NAL_OS_SPEC_PCI_BAR_0_REGISTER_OFFSET;

    do
    {
        /* Checking function arguments */
        if(PDev == NULL || IoResource == NULL)
        {
        }

        /* Searching device by dbsf */
        Device = pci_find_dbsf(DeviceLocation.Pci.Segment,
                               DeviceLocation.Pci.Bus,
                               DeviceLocation.Pci.Device,
                               DeviceLocation.Pci.Function);

        if(Device == NULL)
        {
            NalStatus = NAL_PCICONFIG_NOT_AVAILABLE;
            break;
        }

        /* Enable PCI bus mastering for the device  by setting the PCIM_CMD_BUSMASTEREN bit in the PCIR_COMMAND
         * register */
        Result = pci_enable_busmaster(Device);

        if(Result != 0)
        {
            NalStatus = NAL_PCICONFIG_NOT_AVAILABLE;
            break;
        }

        device_enable(Device);

        /* Read bars types and addresses */
        for(BarIndex=0; BarIndex < 6; BarIndex++ )
        {
            RegisterOffset = PCIR_BAR(BarIndex);
            Register       = pci_read_config(Device, RegisterOffset, sizeof(UINT32));

            if(NAL_OS_SPEC_IS_PCI_MEMORY_BAR(Register))
            {
                IoResource[BarIndex].Type = NAL_IO_TYPE_MEM;
                BarType                   = SYS_RES_MEMORY;
            }
            else
            {
                IoResource[BarIndex].Type = NAL_IO_TYPE_IO;
                BarType                   = SYS_RES_IOPORT;
            }

            IoResource[BarIndex].MemoryAddress = bus_get_resource_start(Device, BarType, RegisterOffset);
        }

        /* Returning device pointer */
        *PDev = Device;

        NalStatus = NAL_SUCCESS;
    } while(0);

    return NalStatus;
}

/***************************************************************************
**
** Name:            _NalFillDeviceResource()
**
** Description:     This fills the I/O and Memory regions for the device.
**                  This provides what is required for I/O to be performed on FreeBSD.
**
** Arguments:       DeviceLocation - Pci location information for this device.
**
** Returns:         IoResource - List of resources.
**                  PDev - device handle in kernel.
**
**                  NAL Status Code. This is the status code for
**                  NalResolveOsSpecificIoctl, not for the individual functions called
**                  from within.
**
****************************************************************************/
NAL_STATUS
_NalFillDeviceResource(
    IN      NAL_DEVICE_LOCATION     DeviceLocation,
    OUT     NAL_IO_RESOURCE*        IoResource,
    OUT     KVOID**                 PDev
    )
{
    return NAL_NOT_IMPLEMENTED;
}

/***************************************************************************
**
** Name:            _NalReleaseRegions()
**
** Description:     This releases the I/O and Memory regions for the device.
**
** Arguments:       PDev - device handle in kernel.
**
** Returns:         NAL Status Code. This is the status code for
**                  NalResolveOsSpecificIoctl, not for the individual functions called
**                  from within.
**
****************************************************************************/
NAL_STATUS
_NalReleaseRegions(
    IN      KVOID*   PDev
    )
{
    NAL_STATUS NalStatus = NAL_INVALID_PARAMETER;

    if(PDev)
    {
        NalStatus = NAL_SUCCESS;
    }

    return NalStatus;
}
