/* SPDX-License-Identifier: BSD-3-Clause */
/*  Copyright (c) 2020, Intel Corporation
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *
 *   3. Neither the name of the Intel Corporation nor the names of its
 *      contributors may be used to endorse or promote products derived from
 *      this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 *  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 *  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 *  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 *  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 *  POSSIBILITY OF SUCH DAMAGE.
 */
/*$FreeBSD$*/

/**
 * @file ice_iov.c
 * @brief Virtualization support functions
 *
 * Contains functions for enabling and managing PCIe virtual function devices,
 * including enabling new VFs, and managing VFs over the virtchnl interface.
 */

#include "ice_iov.h"

/**
 * ice_iov_attach - Initialize SR-IOV PF host support
 * @sc: device softc structure
 *
 * Initialize SR-IOV PF host support at the end of the driver attach process.
 *
 * @pre Must be called from sleepable context (calls malloc() w/ M_WAITOK)
 *
 * @returns 0 if successful, or
 * - ENOMEM if there is no memory for the PF/VF schemas or iov device
 * - ENXIO if the device isn't PCI-E or doesn't support the same SR-IOV
 *   version as the kernel
 * - ENOENT if the device doesn't have the SR-IOV capability
 */
int
ice_iov_attach(struct ice_softc *sc)
{
	device_t dev = sc->dev;
	nvlist_t *pf_schema, *vf_schema;
	int error;

	pf_schema = pci_iov_schema_alloc_node();
	vf_schema = pci_iov_schema_alloc_node();

	pci_iov_schema_add_unicast_mac(vf_schema, "mac-addr", 0, NULL);
	pci_iov_schema_add_bool(vf_schema, "mac-anti-spoof",
	    IOV_SCHEMA_HASDEFAULT, TRUE);
	pci_iov_schema_add_bool(vf_schema, "allow-set-mac",
	    IOV_SCHEMA_HASDEFAULT, FALSE);
	pci_iov_schema_add_bool(vf_schema, "allow-promisc",
	    IOV_SCHEMA_HASDEFAULT, FALSE);
	pci_iov_schema_add_uint16(vf_schema, "num-queues",
	    IOV_SCHEMA_HASDEFAULT, ICE_DEFAULT_VF_QUEUES);

	error = pci_iov_attach(dev, pf_schema, vf_schema);
	if (error != 0) {
		device_printf(dev,
		    "pci_iov_attach failed (error=%s)\n",
		    ice_err_str(error));
		ice_clear_bit(ICE_FEATURE_SRIOV, sc->feat_en);
	} else
		ice_set_bit(ICE_FEATURE_SRIOV, sc->feat_en);

	return (error);
}

/**
 * ice_iov_detach - Teardown SR-IOV PF host support
 * @sc: device softc structure
 *
 * Teardown SR-IOV PF host support at the start of the driver detach process.
 *
 * @returns 0 if successful or IOV support hasn't been setup, or
 * - EBUSY if VFs still exist
 */
int
ice_iov_detach(struct ice_softc *sc)
{
	device_t dev = sc->dev;
	int error;

	error = pci_iov_detach(dev);
	if (error != 0) {
		device_printf(dev,
		    "pci_iov_detach failed (error=%s)\n",
		    ice_err_str(error));
	}

	return (error);
}

/**
 * ice_iov_init - Called by the OS before the first VF is created.
 * @sc: device softc structure
 * @num_vfs: number of VFs to setup resources for
 * @params: configuration parameters for the PF
 *
 * @returns 0 if successful or an error code on failure
 */
int
ice_iov_init(struct ice_softc *sc, uint16_t num_vfs, const nvlist_t *params __unused)
{
	/* Allocate array of VFs, for tracking */
	sc->vfs = (struct ice_vf *)malloc(sizeof(struct ice_vf) * num_vfs, M_ICE, M_NOWAIT |
	    M_ZERO);
	if (sc->vfs == NULL)
		return (ENOMEM);

	/* Initialize each VF with basic information */
	for (int i = 0; i < num_vfs; i++)
		sc->vfs[i].vf_num = i;

	/* Save off number of configured VFs */
	sc->num_vfs = num_vfs;

	return (0);
}

/**
 * ice_iov_add_vf - Called by the OS for each VF to create
 * @sc: device softc structure
 * @vfnum: index of VF to configure
 * @params: configuration parameters for the VF
 *
 * @returns 0 if successful or an error code on failure
 */
int
ice_iov_add_vf(struct ice_softc *sc __unused, uint16_t vfnum __unused, const nvlist_t *params __unused)
{
	return (0);
}

/**
 * ice_iov_uninit - Called by the OS when VFs are destroyed
 * @sc: device softc structure
 */
void
ice_iov_uninit(struct ice_softc *sc)
{
	/* Release memory used for VF tracking */
	if (sc->vfs) {
		free(sc->vfs, M_ICE);
		sc->vfs = NULL;
	}
	sc->num_vfs = 0;
}
