/* SPDX-License-Identifier: BSD-3-Clause */
/*  Copyright (c) 2020, Intel Corporation
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions are met:
 *
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *
 *   3. Neither the name of the Intel Corporation nor the names of its
 *      contributors may be used to endorse or promote products derived from
 *      this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 *  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 *  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 *  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 *  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 *  POSSIBILITY OF SUCH DAMAGE.
 */
/*$FreeBSD$*/

/**
 * @file ice_rdma.h
 * @brief header file for RDMA client interface functions
 *
 * Contains definitions and function calls shared by the ice driver and the
 * RDMA client interface driver.
 *
 * Since these definitions are shared between drivers it is important that any
 * changes are considered carefully for backwards compatibility.
 */
#ifndef _ICE_RDMA_H_
#define _ICE_RDMA_H_

/*
 * The RDMA client interface version is used to help determine
 * incompatibilities between the interface definition shared between the main
 * driver and the client driver.
 *
 * It will follows the semantic version guidelines, that is:
 * Given the version number MAJOR.MINOR.PATCH, increment the:
 *
 * MAJOR version when you make incompatible changes,
 * MINOR version when you add functionality in a backwards-compatible manner, and
 * PATCH version when you make backwards-compatible bug fixes.
 *
 * Any change to this file, or one of the kobject interface files must come
 * with an associated change in one of the MAJOR, MINOR, or PATCH versions,
 * and care must be taken that backwards incompatible changes MUST increment
 * the MAJOR version.
 *
 * Note: Until the MAJOR version is set to at least 1, the above semantic
 * version guarantees may not hold, and this interface should not be
 * considered stable.
 */
#define ICE_RDMA_MAJOR_VERSION 0
#define ICE_RDMA_MINOR_VERSION 1
#define ICE_RDMA_PATCH_VERSION 0

/**
 * @def ICE_RDMA_MAX_MSIX
 * @brief Maximum number of MSI-X vectors that will be reserved
 *
 * Defines the maximum number of MSI-X vectors that an RDMA interface will
 * have reserved in advance. Does not guarantee that many vectors have
 * actually been enabled.
 */
#define ICE_RDMA_MAX_MSIX 64

/**
 * @struct ice_rdma_info
 * @brief RDMA information from the client driver
 *
 * The RDMA client driver will fill in this structure and pass its contents
 * back to the main driver using the ice_rdma_register function.
 *
 * It should fill the version in with the ICE_RDMA_* versions as defined in
 * the ice_rdma.h header.
 *
 * Additionally it must provide a pointer to a kobject class which extends the
 * ice_rdma_di_class with the operations defined in the rdma_if.m interface.
 *
 * If the version specified is not compatible, then the registration will
 * of the RDMA driver will fail.
 */
struct ice_rdma_info {
	uint16_t major_version;
	uint16_t minor_version;
	uint16_t patch_version;

	kobj_class_t rdma_class;
};

#define ICE_RDMA_MAX_USER_PRIORITY	8
#define ICE_RDMA_MAX_MSIX		64

/* Declare the ice_rdma_di kobject class */
DECLARE_CLASS(ice_rdma_di_class);

/**
 * @struct ice_rdma_msix_mapping
 * @brief MSI-X mapping requested by the peer RDMA driver
 *
 * Defines a mapping for MSI-X vectors being requested by the peer RDMA driver
 * for a given PF.
 */
struct ice_rdma_msix_mapping {
	uint8_t itr_indx;
	int aeq_vector;
	int ceq_cnt;
	int *ceq_vector;
};

/**
 * @struct ice_rdma_msix
 * @brief RDMA MSI-X vectors reserved for the peer RDMA driver
 *
 * Defines the segment of the MSI-X vectors for use by the RDMA driver. These
 * are reserved by the PF when it initializes.
 */
struct ice_rdma_msix {
	int base;
	int count;
};

/**
 * @struct ice_rdma_peer
 * @brief RDMA driver information
 *
 * Shared structure used by the RDMA client driver when talking with the main
 * device driver.
 *
 * Because the definition of this structure is shared between the two drivers,
 * its ABI should be handled carefully.
 */
struct ice_rdma_peer {
	/**
	 * The KOBJ_FIELDS macro must come first, in order for it to be used
	 * as a kobject.
	 */
	KOBJ_FIELDS;

	struct ifnet *ifp;
	device_t dev;
	struct resource *pci_mem;
	uint8_t pf_id;
	uint16_t mtu;
	struct ice_rdma_msix msix;
};

int ice_rdma_register(struct ice_rdma_info *info);
int ice_rdma_unregister(void);

#endif
