// SPDX-License-Identifier: GPL-2.0 or Linux-OpenIB
/* Copyright (c) 2019 Intel Corporation */
#include <linux/configfs.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>
#include "main.h"

#if IS_ENABLED(CONFIG_CONFIGFS_FS)
enum irdma_configfs_attr_type {
	IRDMA_ATTR_IW_DCTCP,
	IRDMA_ATTR_IW_TIMELY,
	IRDMA_ATTR_IW_ECN,
	IRDMA_ATTR_ROCE_TIMELY,
	IRDMA_ATTR_ROCE_DCQCN,
	IRDMA_ATTR_ROCE_DCTCP,
	IRDMA_ATTR_ROCE_ENABLE,
	IRDMA_ATTR_RQ_RELAX_ORDER,
	IRDMA_ATTR_IW_OOO,
	IRDMA_ATTR_ENFORCE_SQ_SIZE,
	IRDMA_ATTR_FORCE_FENCE,
	IRDMA_ATTR_ROCE_NO_ICRC,
	IRDMA_ATTR_ENABLE_UP_MAP,
};

struct irdma_vsi_grp {
	struct config_group group;
	struct irdma_device *iwdev;
};

/**
 * irdma_find_iwdev - find a vsi device given a name
 * @name: name of iwdev
 */
struct irdma_device *irdma_find_iwdev(const char *name)
{
#ifdef NETDEV_TO_IBDEV_SUPPORT
	struct ib_device *ibdev = ib_device_get_by_name(name, RDMA_DRIVER_I40IW);

	if (!ibdev)
		return NULL;

	return to_iwdev(ibdev);
#else
	struct irdma_handler *hdl;
	struct list_head *pos;
	struct list_head *tmp;
	struct irdma_device *iwdev;
	unsigned long flags;

	spin_lock_irqsave(&irdma_handler_lock, flags);
	list_for_each_entry (hdl, &irdma_handlers, list) {
		list_for_each_safe (pos, tmp, &hdl->rf.vsi_dev_list) {
			iwdev = container_of(pos, struct irdma_device, list);
			if (!strcmp(name, iwdev->ibdev.name)) {
				spin_unlock_irqrestore(&irdma_handler_lock,
						       flags);
				return iwdev;
			}
		}
	}
	spin_unlock_irqrestore(&irdma_handler_lock, flags);

	return NULL;
#endif /* NETDEV_TO_IBDEV_SUPPORT */
}

/*
 * irdma_configfs_set_vsi_attr - set vsi configfs attribute
 * @item_name: config item name
 * @buf: buffer
 * @irdma_configfs_type_attr: vsi attribute type to set
 */
static int irdma_configfs_set_vsi_attr(struct config_item *item,
				       const char *buf,
				       enum irdma_configfs_attr_type attr_type)
{
	struct irdma_device *iwdev;
	struct irdma_up_info up_map_info = {};
	bool enable;
	int ret = 0;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	if (strtobool(buf, &enable)) {
		ret = -EINVAL;
		goto done;
	}

	switch (attr_type) {
	case IRDMA_ATTR_IW_DCTCP:
		iwdev->iwarp_dctcp_en = enable;
		iwdev->iwarp_ecn_en = !enable;
		break;
	case IRDMA_ATTR_IW_TIMELY:
		iwdev->iwarp_timely_en = enable;
		break;
	case IRDMA_ATTR_IW_ECN:
		iwdev->iwarp_ecn_en = enable;
		break;
	case IRDMA_ATTR_ENABLE_UP_MAP:
		iwdev->up_map_en = enable;
		if (enable) {
			*((u64 *)up_map_info.map) = iwdev->up_up_map;
			up_map_info.use_cnp_up_override = true;
			up_map_info.cnp_up_override = iwdev->cnp_up_override;
		} else {
			*((u64 *)up_map_info.map) = IRDMA_DEFAULT_UP_UP_MAP;
			up_map_info.use_cnp_up_override = false;
		}
		up_map_info.hmc_fcn_idx = iwdev->rf->sc_dev.hmc_fn_id;
		irdma_cqp_up_map_cmd(&iwdev->rf->sc_dev, IRDMA_OP_SET_UP_MAP,
				     &up_map_info);
		break;
	case IRDMA_ATTR_ROCE_NO_ICRC:
		iwdev->roce_no_icrc_en = enable;
		break;
	case IRDMA_ATTR_ROCE_TIMELY:
		iwdev->roce_timely_en = enable;
		break;
	case IRDMA_ATTR_ROCE_DCQCN:
		iwdev->roce_dcqcn_en = enable;
		break;
	case IRDMA_ATTR_ROCE_DCTCP:
		iwdev->roce_dctcp_en = enable;
		break;
	case IRDMA_ATTR_ROCE_ENABLE:
		//rf->roce_en = enable; FIXME: Add when roce/iwarp in configFS
		break;
	case IRDMA_ATTR_IW_OOO:
		iwdev->rf->ooo = enable;
		iwdev->override_ooo = true;
		break;
	case IRDMA_ATTR_RQ_RELAX_ORDER:
		if (enable)
			iwdev->rf->sc_dev.hw_attrs.uk_attrs.feature_flags |=
				IRDMA_FEATURE_RELAX_RQ_ORDER;
		else
			iwdev->rf->sc_dev.hw_attrs.uk_attrs.feature_flags &=
				~IRDMA_FEATURE_RELAX_RQ_ORDER;
		break;
	case IRDMA_ATTR_ENFORCE_SQ_SIZE:
		if (enable)
			iwdev->rf->sc_dev.hw_attrs.uk_attrs.feature_flags |=
				IRDMA_FEATURE_ENFORCE_SQ_SIZE;
		else
			iwdev->rf->sc_dev.hw_attrs.uk_attrs.feature_flags &=
				~IRDMA_FEATURE_ENFORCE_SQ_SIZE;
		break;
	case IRDMA_ATTR_FORCE_FENCE:
		if (enable)
			iwdev->rf->sc_dev.hw_attrs.uk_attrs.feature_flags |=
				IRDMA_FEATURE_FORCE_FENCE;
		else
			iwdev->rf->sc_dev.hw_attrs.uk_attrs.feature_flags &=
				~IRDMA_FEATURE_FORCE_FENCE;
		break;
	default:
		ret = -EINVAL;
	}

done:
	irdma_put_device(iwdev);

	return ret;
}

/**
 * push_mode_show - Show the value of push_mode for device
 * @item: config item
 * @buf: buffer to write to
 */
static ssize_t push_mode_show(struct config_item *item, char *buf)
{
	struct irdma_device *iwdev;
	ssize_t ret;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	ret = sprintf(buf, "%d\n", iwdev->push_mode);

	irdma_put_device(iwdev);

	return ret;
}

/**
 * push_mode_store - Store value for push_mode
 * @item: config item
 * @buf: buf to read from
 * @count: size of buf
 */
static ssize_t push_mode_store(struct config_item *item,
			       const char *buf,
			       size_t count)
{
	struct irdma_device *iwdev;
	bool enable;

	if (strtobool(buf, &enable))
		return -EINVAL;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	iwdev->push_mode = enable;

	irdma_put_device(iwdev);

	return count;
}

/**
 * roce_cwnd_show - Show the value of RoCE cwnd
 * @item: config item
 * @buf: buffer to write to
 */
static ssize_t roce_cwnd_show(struct config_item *item, char *buf)
{
	struct irdma_device *iwdev;
	ssize_t ret;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	ret = sprintf(buf, "%d\n", iwdev->roce_cwnd);

	irdma_put_device(iwdev);

	return ret;
}

/**
 * roce_cwnd_store - Store value for roce_cwnd
 * @item: config item
 * @buf: buf to read from
 * @count: size of buf
 */
static ssize_t roce_cwnd_store(struct config_item *item,
			       const char *buf,
			       size_t count)
{
	struct irdma_device *iwdev;
	u32 rsrc_cwnd;

	if (kstrtou32(buf, 0, &rsrc_cwnd))
		return -EINVAL;

	if (!rsrc_cwnd)
		return -EINVAL;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	iwdev->roce_cwnd = rsrc_cwnd;
	iwdev->override_cwnd = true;

	irdma_put_device(iwdev);

	return count;
}

/*
 * roce_rd_fence_rate_show - Show RoCE read fence rate
 * @item: config item
 * @buf: buffer to write to
 */
static ssize_t roce_rd_fence_rate_show(struct config_item *item, char *buf)
{
	struct irdma_device *iwdev;
	ssize_t ret;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	ret = sprintf(buf, "%d\n", iwdev->rd_fence_rate);

	irdma_put_device(iwdev);

	return ret;
}

/**
 * roce_rd_fence_rate_store - Store RoCE read fence rate
 * @item: config item
 * @buf: buf to read from
 * @count: size of buf
 */
static ssize_t roce_rd_fence_rate_store(struct config_item *item,
					const char *buf, size_t count)
{
	struct irdma_device *iwdev;
	u32 rd_fence_rate;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	if (kstrtou32(buf, 0, &rd_fence_rate))
		return -EINVAL;

	if (!rd_fence_rate || rd_fence_rate > 256)
		return -EINVAL;

	iwdev->rd_fence_rate = rd_fence_rate;
	iwdev->override_rd_fence_rate = true;

	irdma_put_device(iwdev);

	return count;
}

/**
 * roce_cwnd_show - Show the value of RoCE ack_creds
 * @item: config item
 * @buf: buffer to write to
 */
static ssize_t roce_ackcreds_show(struct config_item *item, char *buf)
{
	struct irdma_device *iwdev;
	ssize_t ret;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	ret = sprintf(buf, "%d\n", iwdev->roce_ackcreds);

	irdma_put_device(iwdev);

	return ret;
}

/**
 * roce_ackcreds_store - Store value for roce_ackcreds
 * @item: config item
 * @buf: buf to read from
 * @count: size of buf
 */
static ssize_t roce_ackcreds_store(struct config_item *item,
			       const char *buf,
			       size_t count)
{
	struct irdma_device *iwdev;
	u32 rsrc_ackcreds;

	if (kstrtou32(buf, 0, &rsrc_ackcreds))
		return -EINVAL;

	if (!rsrc_ackcreds || rsrc_ackcreds > 0x1E)
		return -EINVAL;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	iwdev->roce_ackcreds = rsrc_ackcreds;
	iwdev->override_ackcreds = true;

	irdma_put_device(iwdev);

	return count;
}

/**
 * cnp_up_override_store - Store value for CNP override
 * @item: config item
 * @buf: buf to read from
 * @count: size of buf
 */
static ssize_t cnp_up_override_store(struct config_item *item,
				     const char *buf,
				     size_t count)
{
	struct irdma_device *iwdev;
	u8 cnp_override;

	if (kstrtou8(buf, 0, &cnp_override))
		return -EINVAL;

	if (cnp_override > 0x3F)
		return -EINVAL;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	iwdev->cnp_up_override = cnp_override;
	irdma_put_device(iwdev);

	return count;
}

/**
 * cnp_up_override_show - Show value of CNP UP override
 * @item: config item
 * @buf: buffer to write to
 */
static ssize_t cnp_up_override_show(struct config_item *item, char *buf)
{
	struct irdma_device *iwdev;
	ssize_t ret;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	ret = sprintf(buf, "%d\n", iwdev->cnp_up_override);

	irdma_put_device(iwdev);

	return ret;
}

/**
 * up_up_map_store - Store value for UP-UP map
 * @item: config item
 * @buf: buf to read from
 * @count: size of buf
 */
static ssize_t up_up_map_store(struct config_item *item,
			       const char *buf,
			       size_t count)
{
	struct irdma_device *iwdev;
	u64 up_map;

	if (kstrtou64(buf, 0, &up_map))
		return -EINVAL;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	iwdev->up_up_map = up_map;
	irdma_put_device(iwdev);

	return count;
}

/**
 * up_up_map_show - Show value of IP-UP map
 * @item: config item
 * @buf: buffer to write to
 */
static ssize_t up_up_map_show(struct config_item *item, char *buf)
{
	struct irdma_device *iwdev;
	ssize_t ret;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	ret = sprintf(buf, "0x%llx\n", iwdev->up_up_map);

	irdma_put_device(iwdev);

	return ret;
}

/**
 * rsrc_rcv_wnd_show - Show the value of TCP receive window
 * @item: config item
 * @buf: buffer to write to
 */
static ssize_t rcv_wnd_show(struct config_item *item, char *buf)
{
	struct irdma_device *iwdev;
	ssize_t ret;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	ret = sprintf(buf, "%d\n", iwdev->rcv_wnd);

	irdma_put_device(iwdev);

	return ret;
}

/**
 * rsrc_rcv_wnd_store - Store value for rcv_wnd
 * @item: config item
 * @buf: buf to read from
 * @count: size of buf
 */
static ssize_t rcv_wnd_store(struct config_item *item,
			     const char *buf,
			     size_t count)
{
	struct irdma_device *iwdev;
	u32 rsrc_rcv_wnd;

	if (kstrtou32(buf, 0, &rsrc_rcv_wnd))
		return -EINVAL;

	if (rsrc_rcv_wnd < 65536)
		return -EINVAL;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	iwdev->rcv_wnd = rsrc_rcv_wnd;
	iwdev->override_rcv_wnd = true;
	irdma_put_device(iwdev);

	return count;
}

/**
 * rsrc_recv_wnd_show - Show value of TCP receive window scale
 * @item: config item
 * @buf: buffer to write to
 */
static ssize_t rcv_wscale_show(struct config_item *item, char *buf)
{
	struct irdma_device *iwdev;
	ssize_t ret;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	ret = sprintf(buf, "%d\n", iwdev->rcv_wscale);

	irdma_put_device(iwdev);

	return ret;
}

/**
 * rsrc_recv_wscale_store - Store value for recv_wscale
 * @item: config item
 * @buf: buf to read from
 * @count: size of buf
 */
static ssize_t rcv_wscale_store(struct config_item *item,
				const char *buf,
				size_t count)
{
	struct irdma_device *iwdev;
	u8 rsrc_rcv_wscale;

	if (kstrtou8(buf, 0, &rsrc_rcv_wscale))
		return -EINVAL;

	if (rsrc_rcv_wscale > 16)
		return -EINVAL;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	iwdev->rcv_wscale = rsrc_rcv_wscale;

	irdma_put_device(iwdev);

	return count;
}

/**
 * iw_dctcp_enable_show - Show the value of dctcp_enable for vsi
 * @item: config item
 * @buf: buffer to write to
 */
static ssize_t iw_dctcp_enable_show(struct config_item *item, char *buf)
{
	struct irdma_device *iwdev;
	ssize_t ret;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	ret = sprintf(buf, "%d\n", iwdev->iwarp_dctcp_en);

	irdma_put_device(iwdev);

	return ret;
}

/**
 * iw_dctcp_enable_store - Store value of dctcp_enable for vsi
 * @item: config item
 * @buf: buf to read from
 * @count: size of buf
 */
static ssize_t iw_dctcp_enable_store(struct config_item *item,
				  const char *buf,
				  size_t count)
{
	int ret;

	ret = irdma_configfs_set_vsi_attr(item, buf, IRDMA_ATTR_IW_DCTCP);

	if (ret)
		return ret;

	return count;
}

/**
 * iw_ecn_enable_show - Show the value of ecn_enable for vsi
 * @item: config item
 * @buf: buffer to write to
 */
static ssize_t iw_ecn_enable_show(struct config_item *item, char *buf)
{
	struct irdma_device *iwdev;
	ssize_t ret;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	ret = sprintf(buf, "%d\n", iwdev->iwarp_ecn_en);

	irdma_put_device(iwdev);

	return ret;
}

/**
 * iw_ecn_enable_store - Store value of ecn_enable for vsi
 * @item: config item
 * @buf: buf to read from
 * @count: size of buf
 */
static ssize_t iw_ecn_enable_store(struct config_item *item,
				   const char *buf,
				   size_t count)
{
	int ret;

	ret = irdma_configfs_set_vsi_attr(item, buf, IRDMA_ATTR_IW_ECN);

	if (ret)
		return ret;

	return count;
}

/**
 * iw_timely_enable_show - Show value of iwarp_timely_enable for vsi
 * @item: config item
 * @buf: buffer to write to
 */
static ssize_t iw_timely_enable_show(struct config_item *item, char *buf)
{
	struct irdma_device *iwdev;
	ssize_t ret;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	ret = sprintf(buf, "%d\n", iwdev->iwarp_timely_en);

	irdma_put_device(iwdev);

	return ret;
}

/**
 * iw_timely_enable_store - Store value of iwarp_timely_enable for vsi
 * @item: config item
 * @buf: buf to read from
 * @count: size of buf
 */
static ssize_t iw_timely_enable_store(struct config_item *item,
				      const char *buf,
				      size_t count)
{
	int ret;

	ret = irdma_configfs_set_vsi_attr(item, buf, IRDMA_ATTR_IW_TIMELY);

	if (ret)
		return ret;

	return count;
}

/**
 * iw_rtomin_show - Show the value of rtomin for vsi
 * @item: config item
 * @buf: buffer to write to
 */
static ssize_t iw_rtomin_show(struct config_item *item, char *buf)
{
	struct irdma_device *iwdev;
	ssize_t ret;

	iwdev = irdma_find_iwdev(item->ci_name);
	if (!iwdev)
		return -ENODEV;

	ret = sprintf(buf, "%d\n", iwdev->iwarp_rtomin);

	irdma_put_device(iwdev);

	return ret;
}

/**
 * iw_rtomin_store - Store value of iwarp_rtomin for vsi
 * @item: config item
 * @buf: buf to read from
 * @count: size of buf
 */
static ssize_t iw_rtomin_store(struct config_item *item,
			       const char *buf,
			       size_t count)
{
	struct irdma_device *iwdev;
	u8 rtomin;

	if (kstrtou8(buf, 0, &rtomin))
		return -EINVAL;

	iwdev = irdma_find_iwdev(item->ci_name);
	if (!iwdev)
		return -ENODEV;

	iwdev->iwarp_rtomin = rtomin;
	iwdev->override_rtomin = true;

	irdma_put_device(iwdev);

	return count;
}

/**
 * roce_rtomin_show - Show the value of roce_rtomin for vsi
 * @item: config item
 * @buf: buffer to write to
 */
static ssize_t roce_rtomin_show(struct config_item *item, char *buf)
{
	struct irdma_device *iwdev;
	ssize_t ret;

	iwdev = irdma_find_iwdev(item->ci_name);
	if (!iwdev)
		return -ENODEV;

	ret = sprintf(buf, "%d\n", iwdev->roce_rtomin);

	irdma_put_device(iwdev);

	return ret;
}

/**
 * roce_rtomin_store - Store value of roce_rtomin for vsi
 * @item: config item
 * @buf: buf to read from
 * @count: size of buf
 */
static ssize_t roce_rtomin_store(struct config_item *item,
				 const char *buf,
				 size_t count)
{
	struct irdma_device *iwdev;
	u8 rtomin;

	if (kstrtou8(buf, 0, &rtomin))
		return -EINVAL;

	iwdev = irdma_find_iwdev(item->ci_name);
	if (!iwdev)
		return -ENODEV;

	iwdev->roce_rtomin = rtomin;
	iwdev->override_rtomin = true;

	irdma_put_device(iwdev);

	return count;
}

/**
 * roce_timely_enable_show - Show value of roce_timely_enable for vsi
 * @item: config item
 * @buf: buffer to write to
 */
static ssize_t roce_timely_enable_show(struct config_item *item, char *buf)
{
	struct irdma_device *iwdev;
	ssize_t ret;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	ret = sprintf(buf, "%d\n", iwdev->roce_timely_en);

	irdma_put_device(iwdev);

	return ret;
}

/**
 * roce_timely_enable_store - Store value of roce_timely_enable for vsi
 * @item: config item
 * @buf: buf to read from
 * @count: size of buf
 */
static ssize_t roce_timely_enable_store(struct config_item *item,
					const char *buf,
					size_t count)
{
	int ret;

	ret = irdma_configfs_set_vsi_attr(item, buf, IRDMA_ATTR_ROCE_TIMELY);

	if (ret)
		return ret;

	return count;
}

/**
 * roce_no_icrc_enable_show - Show value of no_icrc for vsi
 * @item: config item
 * @buf: buffer to write to
 */
static ssize_t roce_no_icrc_enable_show(struct config_item *item, char *buf)
{
	struct irdma_device *iwdev;
	ssize_t ret;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	ret = sprintf(buf, "%d\n", iwdev->roce_no_icrc_en);

	irdma_put_device(iwdev);

	return ret;
}

/**
 * roce_no_icrc_enable_store - Store value of roce_no_icrc for vsi
 * @item: config item
 * @buf: buf to read from
 * @count: size of buf
 */
static ssize_t roce_no_icrc_enable_store(struct config_item *item,
					 const char *buf,
					 size_t count)
{
	int ret;

	ret = irdma_configfs_set_vsi_attr(item, buf, IRDMA_ATTR_ROCE_NO_ICRC);

	if (ret)
		return ret;

	return count;
}

/**
 * up_map_enable_show - Show value of up_map_enable for PF
 * @item: config item
 * @buf: buffer to write to
 */
static ssize_t up_map_enable_show(struct config_item *item, char *buf)
{
	struct irdma_device *iwdev;
	ssize_t ret;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	ret = sprintf(buf, "%d\n", iwdev->up_map_en);

	irdma_put_device(iwdev);

	return ret;
}

/**
 * up_map_enable_store - Store value of up_map_enable for PF
 * @item: config item
 * @buf: buf to read from
 * @count: size of buf
 */
static ssize_t up_map_enable_store(struct config_item *item,
				   const char *buf,
				   size_t count)
{
	int ret;

	ret = irdma_configfs_set_vsi_attr(item, buf, IRDMA_ATTR_ENABLE_UP_MAP);

	if (ret)
		return ret;

	return count;
}

/**
 * iw_rq_relax_order_enable_show - Show the value of
 * rq_relax_order enable for vsi
 * @item: config item
 * @buf: buffer to write to
 */
static ssize_t rq_relax_order_enable_show(struct config_item *item, char *buf)
{
	struct irdma_device *iwdev;
	ssize_t ret;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	ret = sprintf(buf, "%d\n",
		      (bool)(iwdev->rf->sc_dev.hw_attrs.uk_attrs.feature_flags &
			IRDMA_FEATURE_RELAX_RQ_ORDER));

	irdma_put_device(iwdev);

	return ret;
}

/**
 * iw_rq_relax_order_enable_store - Store rq_relax_order
 * @item: config item
 * @buf: buf to read from
 * @count: size of buf
 */
static ssize_t rq_relax_order_enable_store(struct config_item *item,
					   const char *buf,
					   size_t count)
{
	int ret;

	ret = irdma_configfs_set_vsi_attr(item, buf, IRDMA_ATTR_RQ_RELAX_ORDER);

	if (ret)
		return ret;

	return count;
}

/**
 * sq_enforce_size_enable_show - Show the value of the enforce
 * SQ size
 * @item: config item
 * @buf: buffer to write to
 */
static ssize_t sq_enforce_size_enable_show(struct config_item *item, char *buf)
{
	struct irdma_device *iwdev;
	ssize_t ret;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	ret = sprintf(buf, "%d\n",
		      (bool)(iwdev->rf->sc_dev.hw_attrs.uk_attrs.feature_flags &
			IRDMA_FEATURE_ENFORCE_SQ_SIZE));

	irdma_put_device(iwdev);

	return ret;
}

/**
 * sq_enforce_size_enable_store - Store enforce size
 * @item: config item
 * @buf: buf to read from
 * @count: size of buf
 */
static ssize_t sq_enforce_size_enable_store(struct config_item *item,
					    const char *buf,
					    size_t count)
{
	int ret;

	ret = irdma_configfs_set_vsi_attr(item, buf, IRDMA_ATTR_ENFORCE_SQ_SIZE);

	if (ret)
		return ret;

	return count;
}

/**
 * force_fence_enable_show - Show the value of the force fence
 * @item: config item
 * @buf: buffer to write to
 */
static ssize_t force_fence_enable_show(struct config_item *item, char *buf)
{
	struct irdma_device *iwdev;
	ssize_t ret;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	ret = sprintf(buf, "%d\n",
		      (bool)(iwdev->rf->sc_dev.hw_attrs.uk_attrs.feature_flags &
			IRDMA_FEATURE_FORCE_FENCE));

	irdma_put_device(iwdev);

	return ret;
}

/**
 * force_fence_enable_store - Store force fence 
 * @item: config item
 * @buf: buf to read from
 * @count: size of buf
 */
static ssize_t force_fence_enable_store(struct config_item *item,
					const char *buf,
					size_t count)
{
	int ret;

	ret = irdma_configfs_set_vsi_attr(item, buf, IRDMA_ATTR_FORCE_FENCE);

	if (ret)
		return ret;

	return count;
}

/**
 * iw_ooo_enable_show - Show the value of iw_ooo_enable for vsi
 * @item: config item
 * @buf: buffer to write to
 */
static ssize_t iw_ooo_enable_show(struct config_item *item, char *buf)
{
	struct irdma_device *iwdev;
	ssize_t ret;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	ret = sprintf(buf, "%d\n", iwdev->rf->ooo);

	irdma_put_device(iwdev);

	return ret;
}

/**
 * iw_ooo_enable_store - Store value of iw_ooo_enable for vsi
 * @item: config item
 * @buf: buf to read from
 * @count: size of buf
 */
static ssize_t iw_ooo_enable_store(struct config_item *item,
				   const char *buf,
				   size_t count)
{
	int ret;

	ret = irdma_configfs_set_vsi_attr(item, buf, IRDMA_ATTR_IW_OOO);

	if (ret)
		return ret;

	return count;
}

/**
 * roce_dcqcn_enable_show - Show the value of roce_dcqcn_enable for vsi
 * @item: config item
 * @buf: buffer to write to
 */
static ssize_t roce_dcqcn_enable_show(struct config_item *item, char *buf)
{
	struct irdma_device *iwdev;
	ssize_t ret;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	ret = sprintf(buf, "%d\n", iwdev->roce_dcqcn_en);

	irdma_put_device(iwdev);

	return ret;
}

/**
 * roce_dcqcn_enable_store - Store value of roce_dcqcn_enable for vsi
 * @item: config item
 * @buf: buf to read from
 * @count: size of buf
 */
static ssize_t roce_dcqcn_enable_store(struct config_item *item,
				       const char *buf,
				       size_t count)
{
	int ret;

	ret = irdma_configfs_set_vsi_attr(item, buf, IRDMA_ATTR_ROCE_DCQCN);

	if (ret)
		return ret;

	return count;
}

/* roce_dctcp_enable_show - Show the value of roce_dctcp_enable for vsi
 * @item: config item
 * @buf: buffer to write to
 */
static ssize_t roce_dctcp_enable_show(struct config_item *item, char *buf)
{
	struct irdma_device *iwdev;
	ssize_t ret;

	iwdev = irdma_find_iwdev(item->ci_name);

	if (!iwdev)
		return -ENODEV;

	ret = sprintf(buf, "%d\n", iwdev->roce_dctcp_en);

	irdma_put_device(iwdev);

	return ret;
}

/**
 * roce_dctcp_enable_store - Store value of roce_dctcp_enable for vsi
 * @item: config item
 * @buf: buf to read from
 * @count: size of buf
 */
static ssize_t roce_dctcp_enable_store(struct config_item *item,
                                       const char *buf,
                                       size_t count)
{
	int ret;

	ret = irdma_configfs_set_vsi_attr(item, buf, IRDMA_ATTR_ROCE_DCTCP);

	if (ret)
		return ret;

	return count;
}

CONFIGFS_ATTR(, push_mode);
CONFIGFS_ATTR(, iw_dctcp_enable);
CONFIGFS_ATTR(, iw_timely_enable);
CONFIGFS_ATTR(, iw_ecn_enable);
CONFIGFS_ATTR(, iw_rtomin);
CONFIGFS_ATTR(, rcv_wnd);
CONFIGFS_ATTR(, rcv_wscale);
CONFIGFS_ATTR(, iw_ooo_enable);
CONFIGFS_ATTR(, rq_relax_order_enable);
CONFIGFS_ATTR(, sq_enforce_size_enable);
CONFIGFS_ATTR(, force_fence_enable);
CONFIGFS_ATTR(, up_map_enable);
CONFIGFS_ATTR(, cnp_up_override);
CONFIGFS_ATTR(, up_up_map);
CONFIGFS_ATTR(, roce_timely_enable);
CONFIGFS_ATTR(, roce_no_icrc_enable);
CONFIGFS_ATTR(, roce_dcqcn_enable);
CONFIGFS_ATTR(, roce_dctcp_enable);
CONFIGFS_ATTR(, roce_cwnd);
CONFIGFS_ATTR(, roce_rd_fence_rate);
CONFIGFS_ATTR(, roce_ackcreds);
CONFIGFS_ATTR(, roce_rtomin);

static struct configfs_attribute *irdma_iw_vsi_attrs[] = {
	&attr_push_mode,
	&attr_iw_dctcp_enable,
	&attr_iw_timely_enable,
	&attr_iw_ecn_enable,
	&attr_iw_rtomin,
	&attr_rcv_wnd,
	&attr_rcv_wscale,
	&attr_iw_ooo_enable,
	&attr_rq_relax_order_enable,
	&attr_sq_enforce_size_enable,
	&attr_force_fence_enable,
	&attr_cnp_up_override,
	&attr_up_map_enable,
	&attr_up_up_map,
	NULL,
};

static struct configfs_attribute *irdma_roce_vsi_attrs[] = {
	&attr_push_mode,
	&attr_roce_cwnd,
	&attr_roce_rd_fence_rate,
	&attr_roce_ackcreds,
	&attr_rq_relax_order_enable,
	&attr_sq_enforce_size_enable,
	&attr_force_fence_enable,
	&attr_roce_timely_enable,
	&attr_roce_no_icrc_enable,
	&attr_roce_dcqcn_enable,
	&attr_roce_dctcp_enable,
	&attr_roce_rtomin,
	&attr_cnp_up_override,
	&attr_up_map_enable,
	&attr_up_up_map,
	NULL,
};

static void irdma_release_vsi_grp(struct config_item *item)
{
	struct config_group *group = container_of(item, struct config_group,
						  cg_item);
	struct irdma_vsi_grp *vsi_grp = container_of(group,
						     struct irdma_vsi_grp,
						     group);

	kfree(vsi_grp);
}

static struct configfs_item_operations irdma_vsi_ops = {
	.release = irdma_release_vsi_grp
};

static struct config_item_type irdma_iw_vsi_type = {
	.ct_attrs = irdma_iw_vsi_attrs,
	.ct_item_ops = &irdma_vsi_ops,
	.ct_owner = THIS_MODULE,
};

static struct config_item_type irdma_roce_vsi_type = {
	.ct_attrs = irdma_roce_vsi_attrs,
	.ct_item_ops = &irdma_vsi_ops,
	.ct_owner = THIS_MODULE,
};

/**
 * irdma_vsi_make_group - Creation of subsystem groups
 * @group: config group
 * @name: name of the group
 */
static struct config_group *irdma_vsi_make_group(struct config_group *group,
						 const char *name)
{
	struct irdma_vsi_grp *vsi_grp;
	struct irdma_device *iwdev;

	iwdev = irdma_find_iwdev(name);
	if (!iwdev)
		return ERR_PTR(-ENODEV);

	vsi_grp = kzalloc(sizeof(*vsi_grp), GFP_KERNEL);
	if (!vsi_grp) {
		irdma_put_device(iwdev);
		return ERR_PTR(-ENOMEM);
	}

	vsi_grp->iwdev = iwdev;

	config_group_init(&vsi_grp->group);

	if (iwdev->rf->protocol_used == IRDMA_ROCE_PROTOCOL_ONLY)
		config_group_init_type_name(&vsi_grp->group, name,
					    &irdma_roce_vsi_type);
	else
		config_group_init_type_name(&vsi_grp->group, name,
					    &irdma_iw_vsi_type);

	irdma_put_device(iwdev);

	return &vsi_grp->group;
}

static struct configfs_group_operations irdma_vsi_group_ops = {
	.make_group = irdma_vsi_make_group,
};

static struct config_item_type irdma_subsys_type = {
	.ct_group_ops = &irdma_vsi_group_ops,
	.ct_owner = THIS_MODULE,
};

static struct configfs_subsystem cfs_subsys = {
	.su_group = {
		.cg_item = {
			.ci_namebuf = "irdma",
			.ci_type = &irdma_subsys_type,
		},
	},
};

int irdma_configfs_init(void)
{
	config_group_init(&cfs_subsys.su_group);
	mutex_init(&cfs_subsys.su_mutex);
	return configfs_register_subsystem(&cfs_subsys);
}

void irdma_configfs_exit(void)
{
	configfs_unregister_subsystem(&cfs_subsys);
}
#endif /* CONFIG_CONFIGFS_FS */
