# <copyright>
# INTEL CONFIDENTIAL
#
# Copyright 2022 Intel Corporation
#
# This software and the related documents are Intel copyrighted materials, and your use of
# them is governed by the express license under which they were provided to you ("License").
# Unless the License provides otherwise, you may not use, modify, copy, publish, distribute,
# disclose or transmit this software or the related documents without Intel's prior written
# permission.
#
# This software and the related documents are provided as is, with no express or implied
# warranties, other than those that are expressly stated in the License.
#
# <copyright>

[Diagnostics.CodeAnalysis.SuppressMessageAttribute("PSAvoidUsingPositionalParameters", "", Scope="function")] param()

#.ExternalHelp IntelEthernetCmdlets.dll-Help.xml
function Write-IntelEthernetDebugDump
{
    [CmdletBinding(DefaultParameterSetName = 'ByAdapter')]
    Param(
        [parameter(Mandatory=$true, ParameterSetName="ByName")]
        [ValidateNotNullOrEmpty()]
        [String]
        $Name,

        [parameter(Mandatory=$true, Position=0, ValueFromPipeline=$true, ParameterSetName="ByAdapter")]
        [ValidateNotNullOrEmpty()]
        [Object]
        $Adapter,

        [parameter(Mandatory=$true, ParameterSetName="ByName")]
        [parameter(Mandatory=$true,  Position=1, ParameterSetName="ByAdapter")]
        [SupportsWildcards()]
        [ValidateNotNullOrEmpty()]
        [String[]]
        $Cluster,

        [string]
        [parameter(Mandatory=$false, ParameterSetName="ByName")]
        [parameter(Mandatory=$false, Position=2, ParameterSetName="ByAdapter")]
        [ValidateNotNullOrEmpty()]
        $Path = '',

        [parameter(Mandatory=$false, ParameterSetName="ByName")]
        [parameter(Mandatory=$false, ParameterSetName="ByAdapter")]
        [switch]
        $Force = $false,

        [parameter(Mandatory=$false, ParameterSetName="ByName")]
        [parameter(Mandatory=$false, ParameterSetName="ByAdapter")]
        [switch]
        $Append = $false
    )
    Begin
    {
        $script:ErrorMessagesWrite = @()
        $script:WarningMessagesWrite = @()

        GetIntelEthernetDevices
        $script:SupportedAdapters = $script:SupportedAdapters.Where({$_.Service -in @('icea')})

        $Clusters = $Cluster
    }
    End
    {
        do
        {
            if ($Name -match '\*')
            {
                $script:ErrorMessagesWrite += $Messages.AdapterNotFound -f $Name
                break
            }

            $AdapterName = ValidateSetAdapterNameParams $Name $Adapter ([ref]$script:ErrorMessagesWrite)

            if (-not (ValidateSingleAdapter $input $AdapterName ([ref]$script:ErrorMessagesWrite)))
            {
                break
            }

            $AdapterName = GetSupportedAdapters $AdapterName ([ref]$script:ErrorMessagesWrite)

            if (([string]::IsNullOrEmpty($AdapterName)))
            {
                break
            }

            if (([string]::IsNullOrEmpty($Path)))
            {
                $Timestamp = Get-Date -Format "MMddyyyy"
                $DefaultName = ($AdapterName -replace '\s+', '_') + "_$Timestamp`_dump.bin"
                $UseDefaultPath = $true
            }

            $LogPath = ''
            if (-not (ValidatePathParams ([ref]$LogPath) $UseDefaultPath $DefaultName ([ref]$script:ErrorMessagesWrite)))
            {
                break
            }

            $ClusterIDs = ValidateClusterParam $Clusters $AdapterName
            if ($ClusterIDs.Count)
            {
                $Output = [System.Collections.ArrayList]::new()

                # First Dump the 24 Byte Metadata Cluster
                $Metadata = CreateMetadataCluster $AdapterName
                if ($null -eq $Metadata)
                {
                    foreach ($ClusterID in $ClusterIDs)
                    {
                        $script:ErrorMessagesWrite += $Messages.FailedToReadCluster -f $DebugClusterMap.[int]$ClusterID
                    }
                    break
                }

                $Output.AddRange($Metadata)
                $bWriteFile = $false

                foreach ($ClusterID in $ClusterIDs)
                {
                    $ClusterID = [int]$ClusterID
                    $Dump = ReadCluster $ClusterID
                    if ($Dump)
                    {
                        $Output.AddRange($Dump)
                        $bWriteFile = $true
                        $Dump = $null # hint to Garbage Collector that memory can be freed
                    }
                    else
                    {
                        $script:ErrorMessagesWrite += $Messages.FailedToReadCluster -f $DebugClusterMap.[int]$ClusterID
                    }
                }

                if ($bWriteFile)
                {
                    WriteToFile $LogPath $Output $Append
                }
                $Output = $null # hint to Garbage Collector that memory can be freed
            }
            else
            {
                break
            }

        } while ($false)

        foreach ($WarningMessage in $script:WarningMessagesWrite)
        {
            Write-Warning $WarningMessage
        }

        foreach ($ErrorMessage in $script:ErrorMessagesWrite)
        {
            Write-Error $ErrorMessage
        }
        # trigger Garbage Collection
        [System.GC]::Collect()
    }
}


Function CreateMetadataCluster($AdapterName)
{
    $MetadataClusterOutput = [System.Collections.ArrayList]::new()
    $EETrackIDs = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_EetrackId -Property Id -ErrorAction SilentlyContinue
    if ($EETrackIDs)
    {
        $EETrackID = ($EETrackIDs.Where({$_.InstanceName -eq $AdapterName})).Id

        if ($EETrackID)
        {
            $MetadataClusterOutput.AddRange([System.BitConverter]::GetBytes(0x80000001))  # Metadata Cluster ID
            $MetadataClusterOutput.AddRange([System.BitConverter]::GetBytes($EETrackID))
            $MetadataClusterOutput.AddRange([byte[]]::new(16)) # Reserved / not used
        }
    }

    $MetadataClusterOutput
}


Function WriteToFile($Path, $Output, $Append)
{
    if ($Append)
    {
        Add-Content $Path -Value $Output -Encoding Byte
    }
    else
    {
        Set-Content $Path -Value $Output -Encoding Byte
    }

    Resolve-Path -Path $Path -ErrorAction SilentlyContinue
}


Function ValidateClusterParam($Clusters, $AdapterName)
{
    $ValidClusterIDs = @()
    $InvalidClusters = @()

    # Throw an error for Invalid Clusters, but still run if there are any valid ones
    foreach ($Cluster in $Clusters)
    {
        $ClusterID = $DebugClusterMap.GetEnumerator().Where({$_.Value -like $Cluster}).Name
        if ($null -ne $ClusterID) # Check for corresponding Keys
        {
            $ValidClusterIDs += $ClusterID
            continue
        }

        # Check if they sent in the numerical Cluster ID
        if($Cluster -match "^\d+$")
        {
            if ($DebugClusterMap.[int]$Cluster) # Check for corresponding Values
            {
                $ValidClusterIDs += $Cluster
                continue
            }
        }

        $InvalidClusters += $Cluster
    }

    $InvalidClusters = $InvalidClusters | Sort-Object | Get-Unique
    foreach ($Cluster in $InvalidClusters)
    {
        $script:ErrorMessagesWrite += $Messages.InvalidCluster -f $AdapterName, $Cluster
    }

    $ValidClusterIDs = $ValidClusterIDs | Sort-Object | Get-Unique
    $ValidClusterIDs
}


Function BuildTableOutput($ClusterID, $TableID, $TableIndex, $TableBufferSize, $TableBuffer)
{
    $Output = [System.Collections.ArrayList]::new()

    $Output.AddRange([System.BitConverter]::GetBytes($ClusterID))
    $Output.AddRange([System.BitConverter]::GetBytes($TableID))
    $Output.AddRange([System.BitConverter]::GetBytes($TableBufferSize))
    $Output.AddRange([System.BitConverter]::GetBytes($TableIndex))
    $Output.AddRange([byte[]]::new(8)) # Reserved Bytes
    $Output.AddRange($TableBuffer)

    $Output
}


Function ReadCluster($ClusterID)
{
    $i = 0
    $j = 0

    $TableID = 0
    $TableIndex = 0
    $ClusterOutput = [System.Collections.ArrayList]::new()

    $FinalTableID = 0xFF
    $FinalTableID_FWBug = 0xFFFF
    $FinalTableIndex = [uint32]0xFFFFFFFFL

    while ($TableID -ne $FinalTableID -and $TableID -ne $FinalTableID_FWBug)
    {
        Write-Progress -Activity $Messages.WritingDebugDump -PercentComplete $j
        # Call the OID for the current buffer and the next iteration info
        Write-Verbose "`[IN`] Buffer $i : $($DebugClusterMap.$ClusterID) Cluster `($ClusterID`)`tTableID $TableID`tTableIndex $TableIndex"
        $Dump = GetDump $ClusterID $TableID $TableIndex
        if ($Dump.TableBufferSize)
        {
            $Output = BuildTableOutput $ClusterID $TableID $TableIndex $Dump.TableBufferSize $Dump.TableBuffer
            $ClusterOutput.AddRange($Output)
        }

        if ($TableID -ne $Dump.NextTableID -or
            $TableIndex -eq $FinalTableIndex)
        {
            $TableIndex = 0
            $TableID = $Dump.NextTableID
        }
        else
        {
            $TableIndex = $Dump.NextTableIndex
        }

        $i++

        if ($j++ -ge 100)
        {
            $j = 0
        }
    }

    $ClusterOutput
}


function GetDump($ClusterID, $TableID, $TableIndex)
{
    $TableBuffer = [System.Collections.ArrayList]::new()
    $TableBufferSize = 0

    $MaxOIDBufs = 8
    for ($BufIndex = 0; $BufIndex -lt $MaxOIDBufs; $BufIndex++)
    {
        # Build Input Buffer: BufIndex, ClusterId, TableId, TableIndex starting at the 4th byte
        $InBuf = [byte[]]::new(1024)
        [System.BitConverter]::GetBytes($BufIndex).CopyTo($InBuf, 4)
        [System.BitConverter]::GetBytes($ClusterID).CopyTo($InBuf, 8)
        [System.BitConverter]::GetBytes($TableID).CopyTo($InBuf, 12)
        [System.BitConverter]::GetBytes($TableIndex).CopyTo($InBuf, 16)

        $params = @{
            Type        = [uint32]1
            Version     = [uint32]1
            Size        = [uint32]1048 # Header size (24 bytes) + Data Size (1024)
            Operation   = [uint32]0x22 # Operation = 0x22 is INTEL_GEN_DATA_OP_GET_DUMP
            OpVersion   = [uint32]1
            DataSize    = [uint32]1024
            InBuf       = $InBuf
        }

        # first call WmiSetGenData
        $Result = InvokeCimMethod "IntlLan_SetGenData" $AdapterName "WmiSetGenData" $Params

        # if the Result is INTEL_GEN_DATA_STATUS_CALL_GET_GEN_DATA - read GenData
        if ($Result.OutStatus -eq 1)
        {
            $Result = Get-CimInstance -Namespace "root\wmi" -ClassName IntlLan_GetGenData -Filter "InstanceName = '$AdapterName'"
            if ($null -eq $Result)
            {
                $script:ErrorMessagesWrite += $Messages.FailedToReadCluster -f $DebugClusterMap.[int]$ClusterID
                break
            }

            # DumpHeader consists of (Little Endian):
            #   UInt32 BufIndex
            #   UInt32 TableId
            #   UInt32 TableIndex
            #   UInt32 TableSize
            #   UInt32 FwError
            #   UInt32 BufSize
            #   Byte DumpBuf[512]

            $OIDHeaderSize = 48
            $DumpParametersHeaderSize = 48

            $GenDataLength = $Result.GenData.Length
            $DumpHeaderLength = $GenDataLength - $OIDHeaderSize
            $DumpHeader = [byte[]]::new($DumpHeaderLength)
            $DumpBufLength = $DumpHeaderLength - $DumpParametersHeaderSize
            $DumpBufContent = [byte[]]::new($DumpBufLength)

            # Strip OID response header
            [System.Array]::ConstrainedCopy($Result.GenData, 24, $DumpHeader, 0, $DumpHeaderLength)

            # Strip DumpParameters header
            [System.Array]::ConstrainedCopy($DumpHeader, 24, $DumpBufContent, 0, $DumpBufLength)

            $BufIndex        = [bitconverter]::ToUInt32($DumpHeader, 0)
            $NextTableId     = [bitconverter]::ToUInt32($DumpHeader, 4)
            $NextTableIndex  = [bitconverter]::ToUInt32($DumpHeader, 8)
            $TableSize       = [bitconverter]::ToUInt32($DumpHeader, 12)
            $FwError         = [bitconverter]::ToUInt32($DumpHeader, 16)
            $BufSize         = [bitconverter]::ToUInt32($DumpHeader, 20)

            if ($BufIndex -eq 0)
            {
                Write-Verbose "`[OUT`] $($DebugClusterMap.$ClusterID) Cluster `($ClusterID`)`tNextTableID $($NextTableID)`tNextTableIndex $($NextTableIndex)"

                $TableID = $NextTableID
                $TableIndex = $NextTableIndex
                $TableBufferSize = $TableSize
            }

            if (-Not $BufSize)
            {
                break
            }

            $TableBuffer.AddRange($DumpBufContent[0..($BufSize-1)])
        }
        else
        {
            $script:ErrorMessagesWrite += $Messages.FailedToReadCluster -f $DebugClusterMap.[int]$ClusterID

            Write-Verbose "`[OUT`] Buffer $($BufIndex) : $($DebugClusterMap.$ClusterID) Cluster `($ClusterID`)`tNextTableID $($NextTableID)`tNextTableIndex $($NextTableIndex)`tFwError: $($FwError)"
            break
        }
    }

    return [PsCustomObject] @{
        NextTableId = $TableID
        NextTableIndex = $TableIndex
        TableBuffer = $TableBuffer
        TableBufferSize = $TableBufferSize
    }
}

# SIG # Begin signature block
# MIIotwYJKoZIhvcNAQcCoIIoqDCCKKQCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCEVzikybDpPYsu
# 49GSLw93/fQoV0iteEJ/GpYOveFL8KCCEgUwggWeMIIEhqADAgECAhEAzS1l4rws
# CIvYBjRVawV4ujANBgkqhkiG9w0BAQsFADB8MQswCQYDVQQGEwJHQjEbMBkGA1UE
# CBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYDVQQHEwdTYWxmb3JkMRgwFgYDVQQK
# Ew9TZWN0aWdvIExpbWl0ZWQxJDAiBgNVBAMTG1NlY3RpZ28gUlNBIENvZGUgU2ln
# bmluZyBDQTAeFw0yMTA0MDIwMDAwMDBaFw0yMzA0MDIyMzU5NTlaMIGEMQswCQYD
# VQQGEwJVUzETMBEGA1UECAwKQ2FsaWZvcm5pYTEUMBIGA1UEBwwLU2FudGEgQ2xh
# cmExGjAYBgNVBAoMEUludGVsIENvcnBvcmF0aW9uMRIwEAYDVQQLDAlTb2xhcktN
# Q1MxGjAYBgNVBAMMEUludGVsIENvcnBvcmF0aW9uMIIBojANBgkqhkiG9w0BAQEF
# AAOCAY8AMIIBigKCAYEA7CCN9iKpDHOrRceKhlXFP/tf6Lllw2H2fR9KVI4/fQIx
# MU1hXwnlHmAzMCY7IgcCFY4p3F5/MJGKaqYngwOo28Zo6Q1N6ukysA7PSavmF2RY
# WD6VFeya/2H0PoNeRFjHaRzSeynFFeJAFew9r7UReUwM/507sxZYPQuWWIdAEK7H
# Dqp2VlHmgZOXVGHhNO6GFOKpC/C01g6X3x6OquddRNMt5UrZzZzDo5MpJz9SBB2V
# jiqwZ80dvNR2W2xi90cIHh4BkXvB54UNkp4VTVu16T0k3cweo+C39U7GrCAr5Axz
# DETjBvhNtP1sf9SoRV7xY6g5wssfI7yYT9J0gsifn/Vy8MWH355TPoA+PVhbAu0m
# 9FMz4EWu55nnUurNML2jaUxsos21/7ELat12kWC0tq9fhkODjKO8X9PuiBHflZLk
# d3F4QcSMvuGocWGqE77VV3vn8jlvigm2TOV0CfGTQajGMX0jeTRZ19fzBNkt2X9d
# SSGolI/Kj1gSvCggpkUBAgMBAAGjggGQMIIBjDAfBgNVHSMEGDAWgBQO4TqoUzox
# 1Yq+wbutZxoDha00DjAdBgNVHQ4EFgQUshkNuM2SdwJnW4vFy8c4FtUTrbQwDgYD
# VR0PAQH/BAQDAgeAMAwGA1UdEwEB/wQCMAAwEwYDVR0lBAwwCgYIKwYBBQUHAwMw
# EQYJYIZIAYb4QgEBBAQDAgQQMEoGA1UdIARDMEEwNQYMKwYBBAGyMQECAQMCMCUw
# IwYIKwYBBQUHAgEWF2h0dHBzOi8vc2VjdGlnby5jb20vQ1BTMAgGBmeBDAEEATBD
# BgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsLnNlY3RpZ28uY29tL1NlY3RpZ29S
# U0FDb2RlU2lnbmluZ0NBLmNybDBzBggrBgEFBQcBAQRnMGUwPgYIKwYBBQUHMAKG
# Mmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1JTQUNvZGVTaWduaW5nQ0Eu
# Y3J0MCMGCCsGAQUFBzABhhdodHRwOi8vb2NzcC5zZWN0aWdvLmNvbTANBgkqhkiG
# 9w0BAQsFAAOCAQEAVadLNRW4f/pKMqrbn0BdOoQ8/1EJ87gvVfosei2bLwTEvpmv
# mn2n561H6AFedtIJ6L4FmXII4M4r20i+5LREbI6PpKDmOAf4xW7POxfCRvkTQAZO
# 3zoVxjMQBXo7cZVF1xHCdviXzD1usuIiCF8DLm6z4O/kyeFFNcn816yPQct91Pnk
# SBBVvL+Kwu8xvR+ZIQy632WUA4HnNpRdFnVSzUifEg2GrtsKZR8k+rm2o8K8yjJq
# 3SznwgJQCMVMh3CtRtUwE/c7o/6rvm53fTYJDd3aoPHVgH6S2WqS3+3mQG7A6hTD
# nrP/mYnS4PF7XzxxjZhUlhy4G/MarJPvT9IrNDCCBfUwggPdoAMCAQICEB2iSDBv
# myYY0ILgln0z02owDQYJKoZIhvcNAQEMBQAwgYgxCzAJBgNVBAYTAlVTMRMwEQYD
# VQQIEwpOZXcgSmVyc2V5MRQwEgYDVQQHEwtKZXJzZXkgQ2l0eTEeMBwGA1UEChMV
# VGhlIFVTRVJUUlVTVCBOZXR3b3JrMS4wLAYDVQQDEyVVU0VSVHJ1c3QgUlNBIENl
# cnRpZmljYXRpb24gQXV0aG9yaXR5MB4XDTE4MTEwMjAwMDAwMFoXDTMwMTIzMTIz
# NTk1OVowfDELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFuY2hlc3Rl
# cjEQMA4GA1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSQw
# IgYDVQQDExtTZWN0aWdvIFJTQSBDb2RlIFNpZ25pbmcgQ0EwggEiMA0GCSqGSIb3
# DQEBAQUAA4IBDwAwggEKAoIBAQCGIo0yhXoYn0nwli9jCB4t3HyfFM/jJrYlZilA
# hlRGdDFixRDtsocnppnLlTDAVvWkdcapDlBipVGREGrgS2Ku/fD4GKyn/+4uMyD6
# DBmJqGx7rQDDYaHcaWVtH24nlteXUYam9CflfGqLlR5bYNV+1xaSnAAvaPeX7Wpy
# vjg7Y96Pv25MQV0SIAhZ6DnNj9LWzwa0VwW2TqE+V2sfmLzEYtYbC43HZhtKn52B
# xHJAteJf7wtF/6POF6YtVbC3sLxUap28jVZTxvC6eVBJLPcDuf4vZTXyIuosB69G
# 2flGHNyMfHEo8/6nxhTdVZFuihEN3wYklX0Pp6F8OtqGNWHTAgMBAAGjggFkMIIB
# YDAfBgNVHSMEGDAWgBRTeb9aqitKz1SA4dibwJ3ysgNmyzAdBgNVHQ4EFgQUDuE6
# qFM6MdWKvsG7rWcaA4WtNA4wDgYDVR0PAQH/BAQDAgGGMBIGA1UdEwEB/wQIMAYB
# Af8CAQAwHQYDVR0lBBYwFAYIKwYBBQUHAwMGCCsGAQUFBwMIMBEGA1UdIAQKMAgw
# BgYEVR0gADBQBgNVHR8ESTBHMEWgQ6BBhj9odHRwOi8vY3JsLnVzZXJ0cnVzdC5j
# b20vVVNFUlRydXN0UlNBQ2VydGlmaWNhdGlvbkF1dGhvcml0eS5jcmwwdgYIKwYB
# BQUHAQEEajBoMD8GCCsGAQUFBzAChjNodHRwOi8vY3J0LnVzZXJ0cnVzdC5jb20v
# VVNFUlRydXN0UlNBQWRkVHJ1c3RDQS5jcnQwJQYIKwYBBQUHMAGGGWh0dHA6Ly9v
# Y3NwLnVzZXJ0cnVzdC5jb20wDQYJKoZIhvcNAQEMBQADggIBAE1jUO1HNEphpNve
# aiqMm/EAAB4dYns61zLC9rPgY7P7YQCImhttEAcET7646ol4IusPRuzzRl5ARokS
# 9At3WpwqQTr81vTr5/cVlTPDoYMot94v5JT3hTODLUpASL+awk9KsY8k9LOBN9O3
# ZLCmI2pZaFJCX/8E6+F0ZXkI9amT3mtxQJmWunjxucjiwwgWsatjWsgVgG10Xkp1
# fqW4w2y1z99KeYdcx0BNYzX2MNPPtQoOCwR/oEuuu6Ol0IQAkz5TXTSlADVpbL6f
# ICUQDRn7UJBhvjmPeo5N9p8OHv4HURJmgyYZSJXOSsnBf/M6BZv5b9+If8AjntIe
# Q3pFMcGcTanwWbJZGehqjSkEAnd8S0vNcL46slVaeD68u28DECV3FTSK+TbMQ5Lk
# uk/xYpMoJVcp+1EZx6ElQGqEV8aynbG8HArafGd+fS7pKEwYfsR7MUFxmksp7As9
# V1DSyt39ngVR5UR43QHesXWYDVQk/fBO4+L4g71yuss9Ou7wXheSaG3IYfmm8SoK
# C6W59J7umDIFhZ7r+YMp08Ysfb06dy6LN0KgaoLtO0qqlBCk4Q34F8W2WnkzGJLj
# tXX4oemOCiUe5B7xn1qHI/+fpFGe+zmAEc3btcSnqIBv5VPU4OOiwtJbGvoyJi1q
# V3AcPKRYLqPzW0sH3DJZ84enGm1YMIIGZjCCBE6gAwIBAgITMwAAAES3P/zvWs+i
# egAAAAAARDANBgkqhkiG9w0BAQUFADB/MQswCQYDVQQGEwJVUzETMBEGA1UECBMK
# V2FzaGluZ3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
# IENvcnBvcmF0aW9uMSkwJwYDVQQDEyBNaWNyb3NvZnQgQ29kZSBWZXJpZmljYXRp
# b24gUm9vdDAeFw0xNTA3MjIyMTAzNDlaFw0yNTA3MjIyMTAzNDlaMIGIMQswCQYD
# VQQGEwJVUzETMBEGA1UECBMKTmV3IEplcnNleTEUMBIGA1UEBxMLSmVyc2V5IENp
# dHkxHjAcBgNVBAoTFVRoZSBVU0VSVFJVU1QgTmV0d29yazEuMCwGA1UEAxMlVVNF
# UlRydXN0IFJTQSBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eTCCAiIwDQYJKoZIhvcN
# AQEBBQADggIPADCCAgoCggIBAIASZRc2DsPbCLPQrFcNdu3NJ9NMrVCDYeKqIE0J
# LWQJ3M6Jn8w9qez2z8Hc8dOx1ns3KBErR9o5xrw6GbRfpr19naNjQrZ28qk7K5H4
# 4m/Q7BYgkAk+4uh0yRi0kdRiZNt/owbxiBhqkCI8vP4T8IcUe/bkH47U5FHGEWdG
# CFHLhhRUP7wz/n5snP8WnRi9UY41pqdmyHJn2yFmsdSbeAPAUDrozPDcvJ5M/q8F
# ljUfV1q3/875PbcstvZU3cjnEjpNrkyKt1yatLcgPcp/IjSufjtoZgFE5wFORlOb
# M2D3lL5TN5BzQ/Myw1Pv26r+dE5px2uMYJPexMcM3+EyrsyTO1F4lWeL7j1W/gzQ
# aQ8bD/MlJmszbfduR/pzQ+V+DqVmsSl8MoRjVYnEDcGTVDAZE6zTfTen6106bDVc
# 20HXEtqpSQvf2ICKCZNijrVmzyWIzYS4sT+kOQ/ZAp7rEkyVfPNrBaleFoPMuGfi
# 6BOdzFuC00yz7Vv/3uVzrCM7LQC/NVV0CUnYSVgaf5I25lGSDvMmfRxNF7zJ7EMm
# 0L9BX0CpRET0medXh55QH1dUqD79dGMvsVBlCeZYQi5DGky08CVHWfoEHpPUJkZK
# UIGy3r54t/xnFeHJV4QeD2PW6WK61l9VLupcxigIBCU5uA4rqfJMlxwHPw1S9e3v
# L4IPAgMBAAGjgdAwgc0wEwYDVR0lBAwwCgYIKwYBBQUHAwMwEgYDVR0TAQH/BAgw
# BgEB/wIBAjAdBgNVHQ4EFgQUU3m/WqorSs9UgOHYm8Cd8rIDZsswCwYDVR0PBAQD
# AgGGMB8GA1UdIwQYMBaAFGL7CiFbf0NuEdoJVFBr9dKWcfGeMFUGA1UdHwROMEww
# SqBIoEaGRGh0dHA6Ly9jcmwubWljcm9zb2Z0LmNvbS9wa2kvY3JsL3Byb2R1Y3Rz
# L01pY3Jvc29mdENvZGVWZXJpZlJvb3QuY3JsMA0GCSqGSIb3DQEBBQUAA4ICAQBr
# IpM8PTlUcWRrDvLkPDARxSBKS4YPkvH/M3k62eSYpw5AoCKAfmGy4KcZzyaVMSpl
# 1GpPMYbqwMYuxWSMPUhZzQsvdD2UJhMQQtSXmCdePHbSeGkdGmTnBXJ14OtmQEOf
# jwxG/5dgpshnrRAIm2Km6b46itMHTZ9ykyW8BhHgLJA4Pmcc/RnXnpDOPcLg52Gs
# wOUE9R6ZVAyRDQFWcTeuJ9SeQyKlySfNTeVxEjkkpUFWh/+8VRQPJcqJ7seX5dIT
# /z1+GqCPP8gs16Nw0MdgwPzYPlHnl8Y+O+3PeL6KyuPE8qen7Z6uCAKPoFLbch7V
# O8NNn476m3DH+OO/bD+Sm+Q3PuxqjCn5waK/iz4aaWb7HGNPJgHJAsQ+0v/DQ6gb
# /Zn61LylueKTLzsBxdH0Oi9ow+Bkt1qVXkbMB4NpuzwFklZzNXNFmE582BKlt0Lp
# omP2QmAYcNE7bzHAh8fmceHzRhbp9bhys+ltH2ImSaNJi91ox4toVvfe/PqHJLgD
# gReP5fFnah2u03T3jKVdswuOQimWzknEd35mfAEXGmwUJMOwF3cF2BpAt4Zr2OR7
# QKx+305vJPkggIKMM+fl+inYndqLcF0ryR2CTAtny4RBnucGfhGDRC2KGe70f5rd
# eRw3GR6fP4wpug1cEIY3bEjNRV3NcLy80U1d2MW4djGCFggwghYEAgEBMIGRMHwx
# CzAJBgNVBAYTAkdCMRswGQYDVQQIExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNV
# BAcTB1NhbGZvcmQxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEkMCIGA1UEAxMb
# U2VjdGlnbyBSU0EgQ29kZSBTaWduaW5nIENBAhEAzS1l4rwsCIvYBjRVawV4ujAN
# BglghkgBZQMEAgEFAKBqMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisG
# AQQBgjcCAQsxDjAMBgorBgEEAYI3AgEWMC8GCSqGSIb3DQEJBDEiBCACqkvIDdtS
# gf7D+wTQBICwawknbuKgyYIJjjipblzFnzANBgkqhkiG9w0BAQEFAASCAYC7BotY
# UmB3HczLaBnFrIf3o0NLebEYEiZVT17nRSd57tT0VyVFjQsDOuHO0D5NNN/iUeow
# 25io4eskOmvoDrp6urYH0AKv/96CJI7KXE7PVtXMPqa/Kt9iB7Mei+ariT+YPLKP
# yVkPmy5tWEt0SlQLqN3NgXoRW66z2a17eGNL3ZOW87+Vob+rbW6qGT/QSuwCxM1J
# bk+GgkxWxpwTUFEzpGFCuywHZe0NqoZ4BK3XLDjyF1CnekMxNoc0/dqWPEpVJlYf
# AuGR8DrSCEFWc9BIluznKrhFlVW1YWQFrYo/b9SoZrT7pl6tPXn98FTVeSesrenr
# e62uHZTmlrTwiI5mBRraiGwhp8KSCIef6tM8qLB07NeX2mYY8cFOBnSySEiLPHi+
# Fa4a82QsJy0ltr9pYdCc7/UlZPZjxoGjB7m8DYtPWy6azVKUR3YMIgvLko23a1ga
# z5vGFJ4N0REYrIZfSbd8HL+jOCzPNhfprzULdLZfixNGwIsAouZw3WvDDUyhghNb
# MIITVwYKKwYBBAGCNwMDATGCE0cwghNDBgkqhkiG9w0BBwKgghM0MIITMAIBAzEP
# MA0GCWCGSAFlAwQCAgUAMIH6BgsqhkiG9w0BCRABBKCB6gSB5zCB5AIBAQYKKwYB
# BAGyMQIBATAxMA0GCWCGSAFlAwQCAQUABCDNeOenGRNWhS2WBvzBv/MoVhM4Czlb
# PftaN5T4j/ul4gIVAOqcYSVHU3sss6Q2Lai++El6gwhwGA8yMDIyMTAwNTE3NDYz
# NVoCCFIc7i+faGymoG6kbDBqMQswCQYDVQQGEwJHQjETMBEGA1UECBMKTWFuY2hl
# c3RlcjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMSwwKgYDVQQDDCNTZWN0aWdv
# IFJTQSBUaW1lIFN0YW1waW5nIFNpZ25lciAjM6CCDeowggb2MIIE3qADAgECAhEA
# kDl/mtJKOhPyvZFfCDipQzANBgkqhkiG9w0BAQwFADB9MQswCQYDVQQGEwJHQjEb
# MBkGA1UECBMSR3JlYXRlciBNYW5jaGVzdGVyMRAwDgYDVQQHEwdTYWxmb3JkMRgw
# FgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQxJTAjBgNVBAMTHFNlY3RpZ28gUlNBIFRp
# bWUgU3RhbXBpbmcgQ0EwHhcNMjIwNTExMDAwMDAwWhcNMzMwODEwMjM1OTU5WjBq
# MQswCQYDVQQGEwJHQjETMBEGA1UECBMKTWFuY2hlc3RlcjEYMBYGA1UEChMPU2Vj
# dGlnbyBMaW1pdGVkMSwwKgYDVQQDDCNTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5n
# IFNpZ25lciAjMzCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAJCycT95
# 4dS5ihfMw5fCkJRy7Vo6bwFDf3NaKJ8kfKA1QAb6lK8KoYO2E+RLFQZeaoogNHF7
# uyWtP1sKpB8vbH0uYVHQjFk3PqZd8R5dgLbYH2DjzRJqiB/G/hjLk0NWesfOA9YA
# ZChWIrFLGdLwlslEHzldnLCW7VpJjX5y5ENrf8mgP2xKrdUAT70KuIPFvZgsB3YB
# cEXew/BCaer/JswDRB8WKOFqdLacRfq2Os6U0R+9jGWq/fzDPOgNnDhm1fx9HptZ
# jJFaQldVUBYNS3Ry7qAqMfwmAjT5ZBtZ/eM61Oi4QSl0AT8N4BN3KxE8+z3N0Ofh
# l1tV9yoDbdXNYtrOnB786nB95n1LaM5aKWHToFwls6UnaKNY/fUta8pfZMdrKAza
# rHhB3pLvD8Xsq98tbxpUUWwzs41ZYOff6Bcio3lBYs/8e/OS2q7gPE8PWsxu3x+8
# Iq+3OBCaNKcL//4dXqTz7hY4Kz+sdpRBnWQd+oD9AOH++DrUw167aU1ymeXxMi1R
# +mGtTeomjm38qUiYPvJGDWmxt270BdtBBcYYwFDk+K3+rGNhR5G8RrVGU2zF9OGG
# J5OEOWx14B0MelmLLsv0ZCxCR/RUWIU35cdpp9Ili5a/xq3gvbE39x/fQnuq6xzp
# 6z1a3fjSkNVJmjodgxpXfxwBws4cfcz7lhXFAgMBAAGjggGCMIIBfjAfBgNVHSME
# GDAWgBQaofhhGSAPw0F3RSiO0TVfBhIEVTAdBgNVHQ4EFgQUJS5oPGuaKyQUqR+i
# 3yY6zxSm8eAwDgYDVR0PAQH/BAQDAgbAMAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/
# BAwwCgYIKwYBBQUHAwgwSgYDVR0gBEMwQTA1BgwrBgEEAbIxAQIBAwgwJTAjBggr
# BgEFBQcCARYXaHR0cHM6Ly9zZWN0aWdvLmNvbS9DUFMwCAYGZ4EMAQQCMEQGA1Ud
# HwQ9MDswOaA3oDWGM2h0dHA6Ly9jcmwuc2VjdGlnby5jb20vU2VjdGlnb1JTQVRp
# bWVTdGFtcGluZ0NBLmNybDB0BggrBgEFBQcBAQRoMGYwPwYIKwYBBQUHMAKGM2h0
# dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1JTQVRpbWVTdGFtcGluZ0NBLmNy
# dDAjBggrBgEFBQcwAYYXaHR0cDovL29jc3Auc2VjdGlnby5jb20wDQYJKoZIhvcN
# AQEMBQADggIBAHPa7Whyy8K5QKExu7QDoy0UeyTntFsVfajp/a3Rkg18PTagadnz
# mjDarGnWdFckP34PPNn1w3klbCbojWiTzvF3iTl/qAQF2jTDFOqfCFSr/8R+lmwr
# 05TrtGzgRU0ssvc7O1q1wfvXiXVtmHJy9vcHKPPTstDrGb4VLHjvzUWgAOT4BHa7
# V8WQvndUkHSeC09NxKoTj5evATUry5sReOny+YkEPE7jghJi67REDHVBwg80uIid
# yCLxE2rbGC9ueK3EBbTohAiTB/l9g/5omDTkd+WxzoyUbNsDbSgFR36bLvBk+9uk
# AzEQfBr7PBmA0QtwuVVfR745ZM632iNUMuNGsjLY0imGyRVdgJWvAvu00S6dOHw1
# 4A8c7RtHSJwialWC2fK6CGUD5fEp80iKCQFMpnnyorYamZTrlyjhvn0boXztVoCm
# 9CIzkOSEU/wq+sCnl6jqtY16zuTgS6Ezqwt2oNVpFreOZr9f+h/EqH+noUgUkQ2C
# /L1Nme3J5mw2/ndDmbhpLXxhL+2jsEn+W75pJJH/k/xXaZJL2QU/bYZy06LQwGTS
# OkLBGgP70O2aIbg/r6ayUVTVTMXKHxKNV8Y57Vz/7J8mdq1kZmfoqjDg0q23fbFq
# QSduA4qjdOCKCYJuv+P2t7yeCykYaIGhnD9uFllLFAkJmuauv2AV3Yb1MIIG7DCC
# BNSgAwIBAgIQMA9vrN1mmHR8qUY2p3gtuTANBgkqhkiG9w0BAQwFADCBiDELMAkG
# A1UEBhMCVVMxEzARBgNVBAgTCk5ldyBKZXJzZXkxFDASBgNVBAcTC0plcnNleSBD
# aXR5MR4wHAYDVQQKExVUaGUgVVNFUlRSVVNUIE5ldHdvcmsxLjAsBgNVBAMTJVVT
# RVJUcnVzdCBSU0EgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkwHhcNMTkwNTAyMDAw
# MDAwWhcNMzgwMTE4MjM1OTU5WjB9MQswCQYDVQQGEwJHQjEbMBkGA1UECBMSR3Jl
# YXRlciBNYW5jaGVzdGVyMRAwDgYDVQQHEwdTYWxmb3JkMRgwFgYDVQQKEw9TZWN0
# aWdvIExpbWl0ZWQxJTAjBgNVBAMTHFNlY3RpZ28gUlNBIFRpbWUgU3RhbXBpbmcg
# Q0EwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQDIGwGv2Sx+iJl9AZg/
# IJC9nIAhVJO5z6A+U++zWsB21hoEpc5Hg7XrxMxJNMvzRWW5+adkFiYJ+9UyUnku
# yWPCE5u2hj8BBZJmbyGr1XEQeYf0RirNxFrJ29ddSU1yVg/cyeNTmDoqHvzOWEnT
# v/M5u7mkI0Ks0BXDf56iXNc48RaycNOjxN+zxXKsLgp3/A2UUrf8H5VzJD0BKLwP
# DU+zkQGObp0ndVXRFzs0IXuXAZSvf4DP0REKV4TJf1bgvUacgr6Unb+0ILBgfrhN
# 9Q0/29DqhYyKVnHRLZRMyIw80xSinL0m/9NTIMdgaZtYClT0Bef9Maz5yIUXx7gp
# GaQpL0bj3duRX58/Nj4OMGcrRrc1r5a+2kxgzKi7nw0U1BjEMJh0giHPYla1IXMS
# Hv2qyghYh3ekFesZVf/QOVQtJu5FGjpvzdeE8NfwKMVPZIMC1Pvi3vG8Aij0bdon
# igbSlofe6GsO8Ft96XZpkyAcSpcsdxkrk5WYnJee647BeFbGRCXfBhKaBi2fA179
# g6JTZ8qx+o2hZMmIklnLqEbAyfKm/31X2xJ2+opBJNQb/HKlFKLUrUMcpEmLQTkU
# Ax4p+hulIq6lw02C0I3aa7fb9xhAV3PwcaP7Sn1FNsH3jYL6uckNU4B9+rY5WDLv
# bxhQiddPnTO9GrWdod6VQXqngwIDAQABo4IBWjCCAVYwHwYDVR0jBBgwFoAUU3m/
# WqorSs9UgOHYm8Cd8rIDZsswHQYDVR0OBBYEFBqh+GEZIA/DQXdFKI7RNV8GEgRV
# MA4GA1UdDwEB/wQEAwIBhjASBgNVHRMBAf8ECDAGAQH/AgEAMBMGA1UdJQQMMAoG
# CCsGAQUFBwMIMBEGA1UdIAQKMAgwBgYEVR0gADBQBgNVHR8ESTBHMEWgQ6BBhj9o
# dHRwOi8vY3JsLnVzZXJ0cnVzdC5jb20vVVNFUlRydXN0UlNBQ2VydGlmaWNhdGlv
# bkF1dGhvcml0eS5jcmwwdgYIKwYBBQUHAQEEajBoMD8GCCsGAQUFBzAChjNodHRw
# Oi8vY3J0LnVzZXJ0cnVzdC5jb20vVVNFUlRydXN0UlNBQWRkVHJ1c3RDQS5jcnQw
# JQYIKwYBBQUHMAGGGWh0dHA6Ly9vY3NwLnVzZXJ0cnVzdC5jb20wDQYJKoZIhvcN
# AQEMBQADggIBAG1UgaUzXRbhtVOBkXXfA3oyCy0lhBGysNsqfSoF9bw7J/RaoLlJ
# WZApbGHLtVDb4n35nwDvQMOt0+LkVvlYQc/xQuUQff+wdB+PxlwJ+TNe6qAcJlhc
# 87QRD9XVw+K81Vh4v0h24URnbY+wQxAPjeT5OGK/EwHFhaNMxcyyUzCVpNb0llYI
# uM1cfwGWvnJSajtCN3wWeDmTk5SbsdyybUFtZ83Jb5A9f0VywRsj1sJVhGbks8Vm
# Bvbz1kteraMrQoohkv6ob1olcGKBc2NeoLvY3NdK0z2vgwY4Eh0khy3k/ALWPncE
# vAQ2ted3y5wujSMYuaPCRx3wXdahc1cFaJqnyTdlHb7qvNhCg0MFpYumCf/RoZSm
# Tqo9CfUFbLfSZFrYKiLCS53xOV5M3kg9mzSWmglfjv33sVKRzj+J9hyhtal1H3G/
# W0NdZT1QgW6r8NDT/LKzH7aZlib0PHmLXGTMze4nmuWgwAxyh8FuTVrTHurwROYy
# bxzrF06Uw3hlIDsPQaof6aFBnf6xuKBlKjTg3qj5PObBMLvAoGMs/FwWAKjQxH/q
# EZ0eBsambTJdtDgJK0kHqv3sMNrxpy/Pt/360KOE2See+wFmd7lWEOEgbsausfm2
# usg1XTN2jvF8IAwqd661ogKGuinutFoAsYyr4/kKyVRd1LlqdJ69SK6YMYIELTCC
# BCkCAQEwgZIwfTELMAkGA1UEBhMCR0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFuY2hl
# c3RlcjEQMA4GA1UEBxMHU2FsZm9yZDEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVk
# MSUwIwYDVQQDExxTZWN0aWdvIFJTQSBUaW1lIFN0YW1waW5nIENBAhEAkDl/mtJK
# OhPyvZFfCDipQzANBglghkgBZQMEAgIFAKCCAWswGgYJKoZIhvcNAQkDMQ0GCyqG
# SIb3DQEJEAEEMBwGCSqGSIb3DQEJBTEPFw0yMjEwMDUxNzQ2MzVaMD8GCSqGSIb3
# DQEJBDEyBDDLXqckI9r8lhrwHxyDVJi1IYiLTjWljPyGb/u8uVv+ygq3Vrur1zNW
# zASQ+RFf/YAwge0GCyqGSIb3DQEJEAIMMYHdMIHaMIHXMBYEFKs0ATqsQJcxnwga
# 8LMY4YP4D3iBMIG8BBQC1luV4oNwwVcAlfqI+SPdk3+tjzCBozCBjqSBizCBiDEL
# MAkGA1UEBhMCVVMxEzARBgNVBAgTCk5ldyBKZXJzZXkxFDASBgNVBAcTC0plcnNl
# eSBDaXR5MR4wHAYDVQQKExVUaGUgVVNFUlRSVVNUIE5ldHdvcmsxLjAsBgNVBAMT
# JVVTRVJUcnVzdCBSU0EgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkCEDAPb6zdZph0
# fKlGNqd4LbkwDQYJKoZIhvcNAQEBBQAEggIAKV5jydJpqRbWMjD/fM9T5+F8fDR5
# DjaLwJ4mXMs+CkfoLyR+dvWCmNs5LK9TbyK27JQiaMYuFQL0MlJg3MlRtDAwlnOd
# 87re0X6u0okgX6FDO99WxD/WeNwyzWTI3z1JU/s9K0YedwHg0n7HuQjOeU0QgBCR
# fW4y7kBbKnyMCwbt9OYlzXc7hitaBo8x6Hpk3J8ZAA8joHmZgnx+HEgLCb/L7UvP
# NcMrhaM0TYwCyn4I2pRJO1yCdEsaZ1q4vQPzkwye5Ma8pAIXrtgYWY495QXZCv+0
# acjJLQISdNo6aNTWQOBxn/gIxBmwfVzVlJbNMrxxoWTX1DRE2pETLvc+52f7hD6M
# eEgFIQ3nI7eJqsZN5PUT9ijktE657K4Wq6qmc7CXVGZybsU6Ym7S6tlKBA+vR+Fo
# OqreYBnnnrE35sXIGfIRk6SkWivZXi+6F98h+eIjeNOhnstWLsX33tt3F/BnZdkv
# /9N9QxNFX8jhvwd1zmYNCaSqjsGeZjJ6PAbyOt4Zt+D3GqwiZw6vwy/nXB6AZoLQ
# 1wVUY7SldRh0T0wGuawfvGvcSmcnDvDHx9JlcsmMT3LsTUnir51iU5iYnuU6mnMz
# AaUWv5KboKqGq2xE0toDsBpgijobuvZb+s/+EqRSjk6f+B67QrH3R0YnGy3CDpTy
# foTMmolKYKxL6vw=
# SIG # End signature block
